/*
 * @(#) $Id: BankClient.java,v 1.1.1.1 2003/10/05 18:39:18 pankaj_kumar Exp $
 *
 * Copyright (c) 2002-03 by Pankaj Kumar (http://www.pankaj-k.net). 
 * All rights reserved.
 *
 * The license governing the use of this file can be found in the 
 * root directory of the containing software.
 */
package org.jstk.example.bank.client;

import java.io.InputStreamReader;
import java.io.BufferedReader;
import java.math.BigDecimal;
import java.util.Iterator;
import org.jstk.*;
import org.jstk.example.bank.*;

public class BankClient extends JSTKAbstractTool {
	private static BankIntf bank;
	private static AccountIntf curAcct = null;

	static class OpenCommand extends JSTKCommandAdapter {
		public Object execute(JSTKArgs args) throws JSTKException {
			if (args.getNum() < 1)
				return new JSTKResult(null, false, "No Initial Deposit. Open failed.");
			BigDecimal amt = null;
			try {
				amt = new BigDecimal(args.get(0));
			} catch (NumberFormatException nfe){
				return new JSTKResult(null, false, "Invalid Amount");
			}
			try {
				curAcct = bank.openAccount(amt);
			} catch (Exception e){
				throw new JSTKException("openAccount() failed.", e);
			}
			return new JSTKResult(null, true, "Account Opened: " + curAcct.getAcctNo());
		}
	}
	static class GetCommand extends JSTKCommandAdapter {
		public Object execute(JSTKArgs args) throws JSTKException {
			if (args.getNum() < 1)
				return new JSTKResult(null, false, "No Account No. Get failed.");
			try {
				curAcct = bank.getAccount(args.get(0));
			} catch (Exceptions.AccountNotFound anf){
				return new JSTKResult(null, false, "Not Found. Get failed.");
			}
			return new JSTKResult(null, true, "Current Account: " + curAcct.getAcctNo());
		}
	}
	static class CurrentCommand extends JSTKCommandAdapter {
		public Object execute(JSTKArgs args) throws JSTKException {
			if (curAcct == null)
				return new JSTKResult(null, false, "No Current Account");
			return new JSTKResult(null, true, "Current Account: " + curAcct.getAcctNo());
		}
	}
	static class ListCommand extends JSTKCommandAdapter {
		public Object execute(JSTKArgs args) throws JSTKException {
			Iterator itr = bank.accounts();
			StringBuffer sb = new StringBuffer();
			int idx = 0;
			while (itr.hasNext()){
				AccountIntf acct = (AccountIntf)itr.next();
				sb.append("Account[" + idx + "]: " + acct + "\n");
				++idx;
			}
			return new JSTKResult(null, true, sb.toString());
		}
	}
	static class DepositCommand extends JSTKCommandAdapter {
		public Object execute(JSTKArgs args) throws JSTKException {
			if (curAcct == null)
				return new JSTKResult(null, false, "No Current Account");
			if (args.getNum() < 1)
				return new JSTKResult(null, false, "No Amount Specified");
			BigDecimal amt = null;
			try {
				amt = new BigDecimal(args.get(0));
			} catch (NumberFormatException nfe){
				return new JSTKResult(null, false, "Invalid Amount");
			}
			try {
				curAcct.deposit(amt);
			} catch (Exceptions.AccountClosed ac){
				return new JSTKResult(null, false, "Account Closed");
			}
			return new JSTKResult(null, true, "Deposited: " + amt);
		}
	}
	static class WithdrawCommand extends JSTKCommandAdapter {
		public Object execute(JSTKArgs args) throws JSTKException {
			if (curAcct == null)
				return new JSTKResult(null, false, "No Current Account");
			if (args.getNum() < 1)
				return new JSTKResult(null, false, "No Amount Specified");
			BigDecimal amt = null;
			try {
				amt = new BigDecimal(args.get(0));
			} catch (NumberFormatException nfe){
				return new JSTKResult(null, false, "Invalid Amount");
			}
			try {
				curAcct.withdraw(amt);
			} catch (Exceptions.AccountClosed ac){
				return new JSTKResult(null, false, "Account Closed");
			} catch (Exceptions.InsufficientAmount ac){
				return new JSTKResult(null, false, "Not Enough Funds");
			}
			return new JSTKResult(null, true, "Withdrawn: " + amt);
		}
	}
	static class BalanceCommand extends JSTKCommandAdapter {
		public Object execute(JSTKArgs args) throws JSTKException {
			if (curAcct == null)
				return new JSTKResult(null, false, "No Current Account");
			BigDecimal balance = null;
			try {
				balance = curAcct.getBalance();
			} catch (Exceptions.AccountClosed ac){
				return new JSTKResult(null, false, "Account Closed");
			}
			return new JSTKResult(null, true, "Current Balance: " + balance);
		}
	}
	static class StatementCommand extends JSTKCommandAdapter {
		public Object execute(JSTKArgs args) throws JSTKException {
			if (curAcct == null)
				return new JSTKResult(null, false, "No Current Account");
			return new JSTKResult(null, true, curAcct.getStatement());
		}
	}
	static class QuitCommand extends JSTKCommandAdapter {
		public Object execute(JSTKArgs args) throws JSTKException {
			System.exit(0);
			return null;
		}
	}

	static {
		cmds.put("open", new OpenCommand());
		cmds.put("get", new GetCommand());
		cmds.put("current", new CurrentCommand());
		cmds.put("list", new ListCommand());
		cmds.put("deposit", new DepositCommand());
		cmds.put("withdraw", new WithdrawCommand());
		cmds.put("balance", new BalanceCommand());
		cmds.put("statement", new StatementCommand());
		cmds.put("quit", new QuitCommand());
		cmds.put("exit", new QuitCommand());
	}

	public BankClient(){
		super();

	}

	public String progName(){
		String progName = "java org.jstk.example.bank.BankClient";
		return progName;
	}
	public String briefDescription(){
		return "Client program for bank example";
	}

	public void init(BankIntf bi){
		this.bank = bi;
	}

	public String execCommand(String[] args) throws Exception {
		JSTKOptions opts = new JSTKOptions();
		if (args.length < 1){		// No argument. Print help message.
			return usageString();
		}
		String cmdString = args[0];
		if (cmdString.equals("-h") || cmdString.equals("help") || cmdString.equals("-?")){
			return usageString();
		}

		JSTKCommand cmd = (JSTKCommand)cmds.get(cmdString);
		if (cmd == null){	// Unknown command.
			System.out.println("Unknown Command: " + cmdString);
			return usageString();
		}

		if (args.length > 1 && (args[1].equals("-h") || args[1].equals("help") || args[1].equals("-?"))){
			return cmdUsageString(cmd, cmdString);
		}

		opts.parse(args, 1);

		try {
			JSTKResult result = (JSTKResult)cmd.execute(opts);
			return result.getText();
		} catch (java.security.AccessControlException ace){
			return "Access denied";
		}
	}
}