/*
 * @(#) $Id: Client.java,v 1.1.1.1 2003/10/05 18:39:01 pankaj_kumar Exp $
 *
 * Copyright (c) 2002-03 by Pankaj Kumar (http://www.pankaj-k.net). 
 * All rights reserved.
 *
 * The license governing the use of this file can be found in the 
 * root directory of the containing software.
 */
package rbac;

import java.rmi.RemoteException;
import java.util.Properties;

import javax.ejb.CreateException;
import javax.ejb.RemoveException;
import javax.naming.Context;
import javax.naming.InitialContext;
import javax.naming.NamingException;
import javax.rmi.PortableRemoteObject;
import rba.*;

/**
 * This class illustrates calling a stateless Session Bean and performing
 * the following exercises:
 * <ul>
 * <li> Create a Trader
 * <li> Buy some shares using the Trader
 * <li> Sell some shares using the Trader
 * <li> Remove the Trader
 * </ul>
 *
 * @author Copyright (c) 1998-2002 by BEA Systems, Inc. All Rights Reserved.
 */

public class Client {

  private static final String JNDI_NAME = "rba-TraderHome";

  private String url;
  String username = null;
  String password = null;
  private TraderHome home;

  public Client(String url, String username, String password)
    throws NamingException
  {

    this.url       = url;
    this.username = username;
    this.password = password;

    home = lookupHome();
  }

  /**
   * Runs this example from the command line. Example:
   * <p>
   * <tt>java examples.ejb20.basic.statelessSession.Client "t3://localhost:7001"</tt>
   * <p>
   * The parameters are optional, but if any are supplied,
   * they are interpreted in this order:
   * <p>
   * @param url               URL such as "t3://localhost:7001" of Server
   */
  public static void main(String[] args) throws Exception {

    log("\nBeginning statelessSession.Client...\n");

    String url       = "t3://localhost:7001";
    String username = null;
    String password = null;

    // Parse the argument list
     if (args.length != 3) {
      System.out.println("Usage: java rbac.Client t3://hostname:port username password");
      return;
    } else {
      url = args[0];
      username = args[1];
      password = args[2];
    }

    Client client = null;
    try {
      client = new Client(url, username, password);
    } catch (NamingException ne) {
      System.exit(1);
    }

    try {
      client.example();
    } catch (Exception e) {
      log("There was an exception while creating and using the Trader.");
      log("This indicates that there was a problem communicating with the server: "+e);
    }

    log("\nEnd statelessSession.Client...\n");
  }

  /**
   * Runs this example.
   */
  public void example()
    throws CreateException, RemoteException, RemoveException
  {

    // create a Trader
    log("Creating a trader");
    Trader trader = (Trader) narrow(home.create(), Trader.class);

    String [] stocks = {"BEAS", "MSFT", "AMZN", "HWP" };

      // execute some buys
    for (int i=0; i<stocks.length; i++) {
      int shares = (i+1) * 100;
      log("Buying "+shares+" shares of "+stocks[i]+".");
      trader.buy(stocks[i], shares);
    }

    // execute some sells
    for (int i=0; i<stocks.length; i++) {
      int shares = (i+1) * 100;
      log("Selling "+shares+" shares of "+stocks[i]+".");
      trader.sell(stocks[i], shares);
    }

    // remove the Trader
    log("Removing the trader");
    trader.remove();

  }

  /**
   * RMI/IIOP clients should use this narrow function
   */
  private Object narrow(Object ref, Class c) {
    return PortableRemoteObject.narrow(ref, c);
  }

  /**
   * Lookup the EJBs home in the JNDI tree
   */
  private TraderHome lookupHome()
    throws NamingException
  {
    // Lookup the beans home using JNDI
    Context ctx = getInitialContext();

    try {
      Object home = ctx.lookup(JNDI_NAME);
      return (TraderHome) narrow(home, TraderHome.class);
    } catch (NamingException ne) {
      log("The client was unable to lookup the EJBHome.  Please make sure ");
      log("that you have deployed the ejb with the JNDI name "+JNDI_NAME+" on the WebLogic server at "+url);
      throw ne;
    }
  }

  /**
   * Using a Properties object will work on JDK 1.1.x and Java2
   * clients
   */
  private Context getInitialContext() throws NamingException {

    try {
      // Get an InitialContext
      Properties h = new Properties();
      h.put(Context.INITIAL_CONTEXT_FACTORY,
        "weblogic.jndi.WLInitialContextFactory");
      h.put(Context.PROVIDER_URL, url);

	h.put(Context.SECURITY_AUTHENTICATION, "simple");
	h.put(Context.SECURITY_PRINCIPAL, username);
	h.put(Context.SECURITY_CREDENTIALS, password);

      return new InitialContext(h);
    } catch (NamingException ne) {
      log("We were unable to get a connection to the WebLogic server at "+url);
      log("exception: " + ne);
      log("Please make sure that the server is running.");
      throw ne;
    }
  }

  /**
   * This is the Java2 version to get an InitialContext.
   * This version relies on the existence of a jndi.properties file in
   * the application's classpath.
   *
   */
//    private static Context getInitialContext()
//      throws NamingException
//    {
//      return new InitialContext();
//    }

  private static void log(String s) {
    System.out.println(s);
  }

}
