/*
 * Copyright 2013 Oracle and/or its affiliates.
 * All rights reserved.  You may not modify, use,
 * reproduce, or distribute this software except in
 * compliance with  the terms of the License at:
 * http://developers.sun.com/license/berkeley_license.html
 */


package com.forest.web;

import com.forest.ejb.GroupsBean;
import com.forest.entity.Groups;
import com.forest.web.util.AbstractPaginationHelper;
import com.forest.web.util.JsfUtil;
import com.forest.web.util.PageNavigation;
import java.io.Serializable;
import java.util.ResourceBundle;
import javax.ejb.EJB;
import javax.enterprise.context.SessionScoped;
import javax.faces.component.UIComponent;
import javax.faces.context.FacesContext;
import javax.faces.convert.Converter;
import javax.faces.convert.FacesConverter;
import javax.faces.model.DataModel;
import javax.faces.model.ListDataModel;
import javax.faces.model.SelectItem;
import javax.inject.Named;


@Named(value = "groupsController")
@SessionScoped
public class GroupsController implements Serializable {
    private static final String BUNDLE = "/Bundle";
    private static final long serialVersionUID = 915049365977089806L;
    @EJB
    private com.forest.ejb.GroupsBean ejbFacade;
    private AbstractPaginationHelper pagination;
    private DataModel items = null;
    private Groups current;
    private int selectedItemIndex;

    public GroupsController() {
    }

    public Groups getSelected() {
        if (current == null) {
            current = new Groups();
            selectedItemIndex = -1;
        }

        return current;
    }

    private GroupsBean getFacade() {
        return ejbFacade;
    }

    public AbstractPaginationHelper getPagination() {
        if (pagination == null) {
            pagination = new AbstractPaginationHelper(10) {
                        @Override
                        public int getItemsCount() {
                            return getFacade()
                                       .count();
                        }

                        @Override
                        public DataModel createPageDataModel() {
                            return new ListDataModel(
                                    getFacade().findRange(
                                            new int[] {
                                                getPageFirstItem(),
                                                getPageFirstItem()
                                                + getPageSize()
                                            }));
                        }
                    };
        }

        return pagination;
    }

    public PageNavigation prepareList() {
        recreateModel();

        return PageNavigation.LIST;
    }

    public PageNavigation prepareView() {
        current = (Groups) getItems()
                               .getRowData();
        selectedItemIndex = pagination.getPageFirstItem()
            + getItems()
                  .getRowIndex();

        return PageNavigation.VIEW;
    }

    public PageNavigation prepareCreate() {
        current = new Groups();
        selectedItemIndex = -1;

        return PageNavigation.CREATE;
    }

    public PageNavigation create() {
        try {
            getFacade()
                .create(current);
            JsfUtil.addSuccessMessage(
                    ResourceBundle.getBundle(BUNDLE).getString("GroupsCreated"));

            return prepareCreate();
        } catch (Exception e) {
            JsfUtil.addErrorMessage(
                    e,
                    ResourceBundle.getBundle(BUNDLE).getString(
                            "PersistenceErrorOccured"));

            return null;
        }
    }

    public PageNavigation prepareEdit() {
        current = (Groups) getItems()
                               .getRowData();
        selectedItemIndex = pagination.getPageFirstItem()
            + getItems()
                  .getRowIndex();

        return PageNavigation.EDIT;
    }

    public PageNavigation update() {
        try {
            getFacade()
                .edit(current);
            JsfUtil.addSuccessMessage(
                    ResourceBundle.getBundle(BUNDLE).getString("GroupsUpdated"));

            return PageNavigation.VIEW;
        } catch (Exception e) {
            JsfUtil.addErrorMessage(
                    e,
                    ResourceBundle.getBundle(BUNDLE).getString(
                            "PersistenceErrorOccured"));

            return null;
        }
    }

    public PageNavigation destroy() {
        current = (Groups) getItems()
                               .getRowData();
        selectedItemIndex = pagination.getPageFirstItem()
            + getItems()
                  .getRowIndex();
        performDestroy();
        recreateModel();

        return PageNavigation.LIST;
    }

    public PageNavigation destroyAndView() {
        performDestroy();
        recreateModel();
        updateCurrentItem();

        if (selectedItemIndex >= 0) {
            return PageNavigation.VIEW;
        } else {
            // all items were removed - go back to list
            recreateModel();

            return PageNavigation.LIST;
        }
    }

    private void performDestroy() {
        try {
            getFacade()
                .remove(current);
            JsfUtil.addSuccessMessage(
                    ResourceBundle.getBundle(BUNDLE).getString("GroupsDeleted"));
        } catch (Exception e) {
            JsfUtil.addErrorMessage(
                    e,
                    ResourceBundle.getBundle(BUNDLE).getString(
                            "PersistenceErrorOccured"));
        }
    }

    private void updateCurrentItem() {
        int count = getFacade()
                        .count();

        if (selectedItemIndex >= count) {
            // selected index cannot be bigger than number of items:
            selectedItemIndex = count - 1;

            // go to previous page if last page disappeared:
            if (pagination.getPageFirstItem() >= count) {
                pagination.previousPage();
            }
        }

        if (selectedItemIndex >= 0) {
            current = getFacade()
                          .findRange(
                        new int[] { selectedItemIndex, selectedItemIndex + 1 })
                          .get(0);
        }
    }

    public DataModel getItems() {
        if (items == null) {
            items = getPagination()
                        .createPageDataModel();
        }

        return items;
    }

    private void recreateModel() {
        items = null;
    }

    public PageNavigation next() {
        getPagination()
            .nextPage();
        recreateModel();

        return PageNavigation.LIST;
    }

    public PageNavigation previous() {
        getPagination()
            .previousPage();
        recreateModel();

        return PageNavigation.LIST;
    }

    public SelectItem[] getItemsAvailableSelectMany() {
        return JsfUtil.getSelectItems(
            ejbFacade.findAll(),
            false);
    }

    public SelectItem[] getItemsAvailableSelectOne() {
        return JsfUtil.getSelectItems(
            ejbFacade.findAll(),
            true);
    }

    @FacesConverter(forClass = Groups.class)
    public static class GroupsControllerConverter implements Converter {
        public Object getAsObject(
            FacesContext facesContext,
            UIComponent component,
            String value) {
            if ((value == null) || (value.length() == 0)) {
                return null;
            }

            GroupsController controller = (GroupsController) facesContext.getApplication()
                                                                         .getELResolver()
                                                                         .getValue(
                        facesContext.getELContext(),
                        null,
                        "groupsController");

            return controller.ejbFacade.find(getKey(value));
        }

        java.lang.Integer getKey(String value) {
            java.lang.Integer key;
            key = Integer.valueOf(value);

            return key;
        }

        String getStringKey(java.lang.Integer value) {
            StringBuilder sb = new StringBuilder();
            sb.append(value);

            return sb.toString();
        }

        @Override
        public String getAsString(
            FacesContext facesContext,
            UIComponent component,
            Object object) {
            if (object == null) {
                return null;
            }

            if (object instanceof Groups) {
                Groups o = (Groups) object;

                return getStringKey(o.getId());
            } else {
                throw new IllegalArgumentException(
                        "object " + object + " is of type "
                        + object.getClass().getName() + "; expected type: "
                        + GroupsController.class.getName());
            }
        }
    }
}
