package org.jpwh.test.filtering;

import org.hibernate.ReplicationMode;
import org.hibernate.Session;
import org.hibernate.jdbc.Work;
import org.jpwh.env.DatabaseProduct;
import org.jpwh.env.JPATest;
import org.jpwh.model.filtering.cascade.BankAccount;
import org.jpwh.model.filtering.cascade.Bid;
import org.jpwh.model.filtering.cascade.BillingDetails;
import org.jpwh.model.filtering.cascade.CreditCard;
import org.jpwh.model.filtering.cascade.Item;
import org.jpwh.model.filtering.cascade.User;
import org.testng.annotations.Test;

import javax.persistence.EntityManager;
import javax.persistence.LockTimeoutException;
import javax.persistence.PersistenceException;
import javax.persistence.PessimisticLockException;
import javax.transaction.UserTransaction;
import java.math.BigDecimal;
import java.sql.Connection;
import java.sql.PreparedStatement;
import java.sql.SQLException;
import java.util.concurrent.Callable;
import java.util.concurrent.Executors;

import static org.testng.Assert.*;
import static org.testng.Assert.assertTrue;

public class Cascade extends JPATest {

    @Override
    public void configurePersistenceUnit() throws Exception {
        configurePersistenceUnit("FilteringCascadePU");
    }

    @Test
    public void detachMerge() throws Throwable {
        UserTransaction tx = TM.getUserTransaction();
        try {
            tx.begin();
            EntityManager em = JPA.createEntityManager();

            Long ITEM_ID;
            {
                User user = new User("jandomanski");
                em.persist(user);

                Item item = new Item("Jakiś przedmiot", user);
                em.persist(item);
                ITEM_ID = item.getId();

                Bid firstBid = new Bid(new BigDecimal("99.00"), item);
                item.getBids().add(firstBid);
                em.persist(firstBid);

                Bid secondBid = new Bid(new BigDecimal("100.00"), item);
                item.getBids().add(secondBid);
                em.persist(secondBid);

                em.flush();
            }
            em.clear();

            Item item = em.find(Item.class, ITEM_ID);
            assertEquals(item.getBids().size(), 2); // Inicjuje oferty
            em.detach(item);

            em.clear();

            item.setName("Nowa nazwa");

            Bid bid = new Bid(new BigDecimal("101.00"), item);
            item.getBids().add(bid);

            /* 
               Hibernate scala odłączony obiekt <code>item</code>: Najpierw sprawdza, czy 
               kontekst utrwalania już zawiera obiekt <code>Item</code> z określoną wartością identyfikatora. 
               W tym przypadku nie mamy takiego, dlatego obiekt <code>Item</code> jest ładowany z bazy danych. 
               Hibernate jest na tyle inteigentny, że wie, iż podczas scalania będą potrzebne oferty z 
               kolekcji <code>bids</code>, dlatego pobiera je od razu, za pomocą tej samej kwerendy SQL. 
               Następnie Hibernate kopiuje odłączone wartości <code>item</code> do załadowanego egzemplarza. 
               Są one zwracane w stanie utrwalonym. Ta sama procedura jest stosowana do wszystkich obiektów 
               <code>Bid</code>. Hibernate wykryje, ze jeden z elementów kolekcji <code>bids</code> jest nowy.

             */
            Item mergedItem = em.merge(item);
            // select i.*, b.*
            //  from ITEM i
            //    left outer join BID b on i.ID = b.ITEM_ID
            //  where i.ID = ?

            /* 
               Hibernate utrwalił nowy obiekt <code>Bid</code> podczas scalania. Ten egzemplarz
               ma teraz przypisaną wartość identyfikatora.
             */
            for (Bid b : mergedItem.getBids()) {
                assertNotNull(b.getId());
            }

            /* 
               Podczas synchronizacji kontekstu utrwalania Hibernate wykrywa, że 
               właściwość <code>name</code> obiektu <code>Item</code> zmieniła się podczas scalania.
               Nowy obiekt <code>Bid</code> także zostanie zapisany.
             */
            em.flush();
            // update ITEM set NAME = ? where ID = ?
            // insert into BID values (?, ?, ?, ...)

            em.clear();

            item = em.find(Item.class, ITEM_ID);
            assertEquals(item.getName(), "Nowa nazwa");
            assertEquals(item.getBids().size(), 3);

            tx.commit();
            em.close();

        } finally {
            TM.rollback();
        }
    }

    @Test
    public void refresh() throws Throwable {
        UserTransaction tx = TM.getUserTransaction();
        try {
            tx.begin();
            EntityManager em = JPA.createEntityManager();

            Long USER_ID;
            Long CREDIT_CARD_ID = null;
            {

                User user = new User("jandomanski");
                user.getBillingDetails().add(
                    new CreditCard("Jan Domanski", "1234567890", "11", "2020")
                );
                user.getBillingDetails().add(
                    new BankAccount("Jan Domanski", "45678", "Some Bank", "1234")
                );
                em.persist(user);
                em.flush();

                USER_ID = user.getId();
                for (BillingDetails bd : user.getBillingDetails()) {
                    if (bd instanceof CreditCard)
                        CREDIT_CARD_ID = bd.getId();
                }
                assertNotNull(CREDIT_CARD_ID);
            }
            tx.commit();
            em.close();
            // Blokady z instrukcji INSERT muszą być zwolnione. Zatwierdzamy transakcję i zaczynamy nową jednostkę pracy

            tx.begin();
            em = JPA.createEntityManager();

            /* 
               Egzemplarz obiektu <code>User</code> jest ładowany z bazy danych.
             */
            User user = em.find(User.class, USER_ID);

            /* 
               Leniwa kolekcja <code>billingDetails</code> jest inicjowana w momencie
               iterowania po elementach, albo w chwili wywołania metody <code>size()</code>.
             */
            assertEquals(user.getBillingDetails().size(), 2);
            for (BillingDetails bd : user.getBillingDetails()) {
                assertEquals(bd.getOwner(), "Jan Domanski");
            }

            // Ktoś zmodyfikował informacje rozliczeniowe w bazie danych!
            final Long SOME_USER_ID = USER_ID;
            final Long SOME_CREDIT_CARD_ID = CREDIT_CARD_ID;
            // Osobna transakcja, dlatego nie ma blokad w bazie danych na
            // zaktualizowanych /usuniętych wierszach. Możemy wykonać dla nich operację SELECT 
            //ponownie w pierwotnej transakcji
            Executors.newSingleThreadExecutor().submit(new Callable<Object>() {
                @Override
                public Object call() throws Exception {

                    UserTransaction tx = TM.getUserTransaction();
                    try {
                        tx.begin();
                        EntityManager em = JPA.createEntityManager();

                        em.unwrap(Session.class).doWork(new Work() {
                            @Override
                            public void execute(Connection con) throws SQLException {
                                PreparedStatement ps;

                                /* Usunięie karty kredytowej. To spowoduje, że odświeżanie zawiedzie
                                   i zostanie zgłoszony wyjątek EntityNotFoundException!
                                ps = con.prepareStatement(
                                    "delete from CREDITCARD where ID = ?"
                                );
                                ps.setLong(1, SOME_CREDIT_CARD_ID);
                                ps.executeUpdate();
                                ps = con.prepareStatement(
                                    "delete from BILLINGDETAILS where ID = ?"
                                );
                                ps.setLong(1, SOME_CREDIT_CARD_ID);
                                ps.executeUpdate();
                                */

                                // Aktualizacja konta bankowego
                                ps = con.prepareStatement(
                                    "update BILLINGDETAILS set OWNER = ? where USER_ID = ?"
                                );
                                ps.setString(1, "Domanski Jan");
                                ps.setLong(2, SOME_USER_ID);
                                ps.executeUpdate();
                            }
                        });

                        tx.commit();
                        em.close();
                    } catch (Exception ex) {
                        // This should NOT fail
                        TM.rollback();
                    }
                    return null;
                }
            }).get();


            /* 
               W chwili wywołania <code>refresh()</code> dla zarządzanego egzemplarza <code>User</code> 
               Hibernate kaskadowo wykonuje operację dla zarządzanego obiektu <code>BillingDetails</code> 
               i odświeża wszystkie za pomocą instrukcji SQL <code>SELECT</code>. 
               Jeśli jednego z tych egzemplarzy nie ma już w bazie danych, to Hibernate zgłasza wyjątek 
               <code>EntityNotFoundException</code>.Następnie Hibernate odświeża egzemplarz
               <code>User</code> i w sposób zachłanny ładuje całą kolekcję 
               <code>billingDetails</code>, żeby wykryć nowe obiekty <code>BillingDetails</code>.

             */
            em.refresh(user);
            // select * from CREDITCARD join BILLINGDETAILS where ID = ?
            // select * from BANKACCOUNT join BILLINGDETAILS where ID = ?
            // select * from USERS
            //  left outer join BILLINGDETAILS
            //  left outer join CREDITCARD
            //  left outer JOIN BANKACCOUNT
            // where ID = ?

            for (BillingDetails bd : user.getBillingDetails()) {
                assertEquals(bd.getOwner(), "Domanski Jan");
            }

            tx.commit();
            em.close();

        } finally {
            TM.rollback();
        }
    }

    @Test
    public void replicate() throws Throwable {
        UserTransaction tx = TM.getUserTransaction();
        try {
            Long ITEM_ID;
            Long USER_ID;

            {
                tx.begin();
                EntityManager em = JPA.createEntityManager();

                User user = new User("jandomanski");
                em.persist(user);
                USER_ID = user.getId();

                Item item = new Item("Jakiś przedmiot", user);
                em.persist(item);
                ITEM_ID = item.getId();

                tx.commit();
                em.close();
            }

            tx.begin();
            EntityManager em = JPA.createEntityManager();

            Item item = em.find(Item.class, ITEM_ID);

            // Inicjalizacja leniwej właściwości Item#seller
            assertNotNull(item.getSeller().getUsername());

            tx.commit();
            em.close();

            tx.begin();
            EntityManager otherDatabase = // ... pobierz EntityManager
                JPA.createEntityManager();

            otherDatabase.unwrap(Session.class)
                .replicate(item, ReplicationMode.OVERWRITE);
            // select ID from ITEM where ID = ?
            // select ID from USERS where ID = ?

            tx.commit();
            // update ITEM set NAME = ?, SELLER_ID = ?, ... where ID = ?
            // update USERS set USERNAME = ?, ... where ID = ?
            otherDatabase.close();

        } finally {
            TM.rollback();
        }
    }

}
