package org.jpwh.test.concurrency;

import org.hibernate.Session;
import org.hibernate.jdbc.Work;
import org.jpwh.env.DatabaseProduct;
import org.jpwh.model.concurrency.version.Category;
import org.jpwh.model.concurrency.version.Item;
import org.testng.annotations.Test;

import javax.persistence.EntityManager;
import javax.persistence.LockModeType;
import javax.persistence.LockTimeoutException;
import javax.persistence.PersistenceException;
import javax.persistence.PessimisticLockException;
import javax.persistence.PessimisticLockScope;
import javax.transaction.UserTransaction;
import java.math.BigDecimal;
import java.sql.Connection;
import java.sql.SQLException;
import java.util.HashMap;
import java.util.List;
import java.util.Map;
import java.util.concurrent.Callable;
import java.util.concurrent.Executors;

import static org.testng.Assert.assertEquals;
import static org.testng.Assert.assertTrue;

public class Locking extends Versioning {

    @Test
    public void pessimisticReadWrite() throws Exception {
        final ConcurrencyTestData testData = storeCategoriesAndItems();
        Long[] CATEGORIES = testData.categories.identifiers;

        UserTransaction tx = TM.getUserTransaction();
        try {
            tx.begin();
            EntityManager em = JPA.createEntityManager();

            BigDecimal totalPrice = new BigDecimal(0);
            for (Long categoryId : CATEGORIES) {

               /* 
                   Dla każdej <code>Kategorii</code> odpytaj o wszystkie egzemplarze <code>Item</code> w trybie
                   blokady <code>PESSIMISTIC_READ</code>. Hibernate zablokuje wiersze w 
                   bazie danych za pomocą kwerendy SQL. Jeśli to możliwe, odczekaj 5 sekund na wypadek gdyby jakaś
                   inna transakcja już utrzymywała kolidującą blokadę.  Jeśli nie można ustanowić 
                   blokady, to zapytanie zgłasza wyjątek.
                 */

                List<Item> items =
                    em.createQuery("select i from Item i where i.category.id = :catId")
                        .setLockMode(LockModeType.PESSIMISTIC_READ)
                        .setHint("javax.persistence.lock.timeout", 5000)
                        .setParameter("catId", categoryId)
                        .getResultList();

                 /* 
                   Jeśli kwerenda pomyślnie zwróci sterowanie, to jest to sygnał, ze mamy blokadę na wyłączność 
                   na danych. Żadna inna transakcja nie będzie mogła uzyskać blokady na wyłączność, 
                   ani nie będzie mogła zmodyfikować danych aż do zatwierdzenia transakcji.
                 */

                for (Item item : items)
                    totalPrice = totalPrice.add(item.getBuyNowPrice());

                // Teraz współbieżna transakcja będzie starała się uzyskać blokadę zapisu. Próba nie powiedzie się, ponieważ
                // utrzymujemy już blokadę na odczyt danych. Zwróćmy uwagę, ze w przypadku bazy danych H2 właściwie nie ma 
                // blokad na odczyt lub zapis. Są jedynie blokady na wyłączność.
                if (categoryId.equals(testData.categories.getFirstId())) {
                    Executors.newSingleThreadExecutor().submit(new Callable<Object>() {
                        @Override
                        public Object call() throws Exception {
                            UserTransaction tx = TM.getUserTransaction();
                            try {
                                tx.begin();
                                EntityManager em = JPA.createEntityManager();

                               // Próba blokady następnej kwerendy musi zawieść w pewnym momencie.
                                //Chcemy odczekać 5 sekund, aż blokada stanie się dostępna
                                //
                                // - H2 zawodzi z domyślnym, globalnym timeoutem wynoszącym 1 sekundę.
                                //
                                // - Oracle wspiera dynamiczne timeouty blokad, ustawia się je za pomocą
                                //   podpowiedzi 'javax.persistence.lock.timeout' dla kwerendy:
                                //
                                //      brak podpowiedzi == FOR UPDATE
                                //      javax.persistence.lock.timeout 0ms == FOR UPDATE NOWAIT
                                //      javax.persistence.lock.timeout >0ms == FOR UPDATE WAIT [sekundy]
                                //
                                // - PostgreSQL nie wspiera timeoutów i po prostu zawiesza się, w przypadku gdy dla kwerendy
                                //   nie zostanie określone NOWAIT. Innym możliwym sposobem
                                //   czekania na blokadę jest ustawienie timeoutu instrukcji dla całego
                                //   połączenia (sesji).
                                if (TM.databaseProduct.equals(DatabaseProduct.POSTGRESQL)) {
                                    em.unwrap(Session.class).doWork(new Work() {
                                        @Override
                                        public void execute(Connection connection) throws SQLException {
                                            connection.createStatement().execute("set statement_timeout = 5000");
                                        }
                                    });
                                }
                               // - MySQL także nie wspiera timeoutów blokady zapytania, ale można
                                //   ustawić timeout dla całego połączenia (sesji).
                                if (TM.databaseProduct.equals(DatabaseProduct.MYSQL)) {
                                    em.unwrap(Session.class).doWork(new Work() {
                                        @Override
                                        public void execute(Connection connection) throws SQLException {
                                            connection.createStatement().execute("set innodb_lock_wait_timeout = 5;");
                                        }
                                    });
                                }

                                // Przeniesienie pierwszego elementu z pierwszej kategorii do ostatniej kategorii
                                // Ta kwerenda powinna zawieść, kiedy ktoś inny ustanowi blokadę na poziomie wierszy. 
                                List<Item> items =
                                    em.createQuery("select i from Item i where i.category.id = :catId")
                                        .setParameter("catId", testData.categories.getFirstId())
                                        .setLockMode(LockModeType.PESSIMISTIC_WRITE) // Zapobiega równoległemu dostępowi
                                        .setHint("javax.persistence.lock.timeout", 5000) // Działa tylko dla Oracle...
                                        .getResultList();

                                Category lastCategory = em.getReference(
                                    Category.class, testData.categories.getLastId()
                                );

                                items.iterator().next().setCategory(lastCategory);

                                tx.commit();
                                em.close();
                            } catch (Exception ex) {
                                // To powinno się nie powieść. Dane są już zablokowane!
                                TM.rollback();

                                if (TM.databaseProduct.equals(DatabaseProduct.POSTGRESQL)) {
                                    // Timeout na poziomie instrukcji dla PostgreSQL nie generuje specyficznego wyjątku
                                    assertTrue(ex instanceof PersistenceException);
                                } else if (TM.databaseProduct.equals(DatabaseProduct.MYSQL)) {
                                    // W MySQL zgłaszany jest wyjątek LockTimeoutException
                                    assertTrue(ex instanceof LockTimeoutException);
                                } else {
                                    // Dla H2 i Oracle zgłaszany jest wyjątek PessimisticLockException
                                    assertTrue(ex instanceof PessimisticLockException);
                                }
                            }
                            return null;
                        }
                    }).get();
                }
            }

            /* 
               Blokady zostaną zwolnione po zatwierdzeniu transakcji.
             */
            tx.commit();
            em.close();

            assertEquals(totalPrice.compareTo(new BigDecimal("108")), 0);
        } finally {
            TM.rollback();
        }
    }

    @Test
    public void findLock() throws Exception {
        final ConcurrencyTestData testData = storeCategoriesAndItems();
        Long CATEGORY_ID = testData.categories.getFirstId();

        UserTransaction tx = TM.getUserTransaction();
        try {
            tx.begin();
            EntityManager em = JPA.createEntityManager();

            Map<String, Object> hints = new HashMap<String, Object>();
            hints.put("javax.persistence.lock.timeout", 5000);

            // Uruchamia SELECT .. FOR UPDATE WAIT 5000 jeśli wspiera to dialekt
            Category category =
                em.find(
                    Category.class,
                    CATEGORY_ID,
                    LockModeType.PESSIMISTIC_WRITE,
                    hints
                );

            category.setName("Nowa nazwa");

            tx.commit();
            em.close();
        } finally {
            TM.rollback();
        }
    }

    // TODO: Te testy zawiodą, ponieważ nullowalne zewnętrznie złączone krotki nie mogą być blokowane w bazie Postgres
    @Test(groups = {"H2", "MYSQL", "ORACLE"})
    public void extendedLock() throws Exception {
        final ConcurrencyTestData testData = storeCategoriesAndItems();
        Long ITEM_ID = testData.items.getFirstId();

        UserTransaction tx = TM.getUserTransaction();
        try {
            tx.begin();
            EntityManager em = JPA.createEntityManager();

            Map<String, Object> hints = new HashMap<String, Object>();
            hints.put("javax.persistence.lock.scope", PessimisticLockScope.EXTENDED);

            Item item =
                em.find(
                    Item.class,
                    ITEM_ID,
                    LockModeType.PESSIMISTIC_READ,
                    hints
                );

            // TODO Ta kwerenda ładująca zdjęcia powinna zablokować wiersze zdjęć, a tego nie robi.
            assertEquals(item.getImages().size(), 0);

            item.setName("Nowa nazwa");

            tx.commit();
            em.close();
        } finally {
            TM.rollback();
        }
    }
}
