package org.jpwh.helloworld;

import org.jpwh.env.TransactionManagerTest;
import org.jpwh.model.helloworld.Message;
import org.testng.annotations.Test;

import javax.persistence.EntityManager;
import javax.persistence.EntityManagerFactory;
import javax.persistence.Persistence;
import javax.transaction.UserTransaction;
import java.util.List;

import static org.testng.Assert.assertEquals;

public class HelloWorldJPA extends TransactionManagerTest {

    @Test
    public void storeLoadMessage() throws Exception {

        EntityManagerFactory emf =
            Persistence.createEntityManagerFactory("HelloWorldPU");

        try {
            {
                /* 
                    Uzyskanie dostępu do standardowego API transakcji <code>UserTransaction</code> i
                    rozpoczęcie transakcji dla tego wątku wykonania
                 */
                UserTransaction tx = TM.getUserTransaction();
                tx.begin();

                /* 
                    Rozpoczęcie nowej sesji z bazą danych poprzez utworzenie obiektu <code>EntityManager</code>. 
                    Będzie to kontekst dla wszystkich operacji utrwalania.
                 */
                EntityManager em = emf.createEntityManager();

                /* 
                    Utworzenie nowego egzemplarza zmapowanej klasy modelu dziedziny <code>Message</code> i ustawienie jej właściwości <code>text</code>.

                 */
                Message message = new Message();
                message.setText("Witaj świecie!");

                /* 
                    Pobranie przejściowego egzemplarza do kontekstu utrwalania. Utrwalenie egzemplarza.
                    Hibernate wie teraz, że chcemy zapisać te dane, ale niekoniecznie wywołuje bazę danych natychmiast.
                 */
                em.persist(message);

                /* 
                    Zatwierdzenie transakcji. Hibernate sprawdza teraz automatycznie kontekst utrwalania i
                    uruchamia potrzebną instrukcję SQL  <code>INSERT</code>.
                 */
                tx.commit();
                // INSERT into MESSAGE (ID, TEXT) values (1, 'Witaj świecie!')

                /* 
                    Jeżeli stworzyliśmy obiekt <code>EntityManager</code>, to teraz trzeba go zamknąć.
                 */
                em.close();
            }

            {
                /* 
                    Każda interakcja z bazą danych powinna się odbywać w obrębie jawnych granic transakcji,
                    nawet jeśli tylko czytamy dane.
                 */
                UserTransaction tx = TM.getUserTransaction();
                tx.begin();

                EntityManager em = emf.createEntityManager();

                /* 
                    Uruchomienie zapytania w celu pobrania wszystkich egzemplarzy klasy <code>Message</code> z bazy danych.
                 */
                List<Message> messages =
                    em.createQuery("select m from Message m").getResultList();
                // SELECT * from MESSAGE


                assertEquals(messages.size(), 1);
                assertEquals(messages.get(0).getText(), "Witaj świecie!");

                /* 
                    Wartość właściwości można zmienić. Hibernate wykryje to automatycznie, ponieważ
                    załadowany <code>Message</code> nadal jest dołączony do kontekstu utrwalania, do którego został załadowany.
                 */
                messages.get(0).setText("Zabierz mnie do lidera!");

                /* 
                    W chwili zatwierdzania framework Hibernate sprawdza kontekst utrwalania w poszukiwaniu „zabrudzonego” stanu, a następnie automatycznie uruchamia 
                    instrukcję SQL <code>UPDATE</code> w celu zsynchronizowania wartości w pamięci ze stanem bazy danych.
                 */
                tx.commit();
                // UPDATE MESSAGE set TEXT = Zabierz mnie do lidera!' where ID = 1
                em.close();
            }

        } finally {
            TM.rollback();
            emf.close();
        }
    }

}

