package org.jpwh.test.fetching;

import org.hibernate.Hibernate;
import org.hibernate.proxy.HibernateProxyHelper;
import org.jpwh.env.JPATest;
import org.jpwh.model.fetching.proxy.Bid;
import org.jpwh.model.fetching.proxy.Category;
import org.jpwh.model.fetching.proxy.Item;
import org.jpwh.model.fetching.proxy.User;
import org.jpwh.shared.util.CalendarUtil;
import org.jpwh.shared.util.TestData;
import org.testng.annotations.Test;

import javax.persistence.EntityManager;
import javax.persistence.Persistence;
import javax.persistence.PersistenceUtil;
import javax.transaction.UserTransaction;

import java.math.BigDecimal;
import java.util.HashSet;
import java.util.Set;

import static org.testng.Assert.*;

public class LazyProxyCollections extends JPATest {

    @Override
    public void configurePersistenceUnit() throws Exception {
        configurePersistenceUnit("FetchingProxyPU");
    }

    public FetchTestData storeTestData() throws Exception {
        UserTransaction tx = TM.getUserTransaction();
        tx.begin();
        EntityManager em = JPA.createEntityManager();

        Long[] categoryIds = new Long[3];
        Long[] itemIds = new Long[3];
        Long[] userIds = new Long[3];

        User jandomanski = new User("jandomanski");
        em.persist(jandomanski);
        userIds[0] = jandomanski.getId();

        User janinadomanska = new User("janinadomanska");
        em.persist(janinadomanska);
        userIds[1] = janinadomanska.getId();

        User robertdomanski = new User("robertdomanski");
        em.persist(robertdomanski);
        userIds[2] = robertdomanski.getId();

        Category category = new Category("Kategoria pierwsza");
        em.persist(category);
        categoryIds[0] = category.getId();

        Item item = new Item("Przedmiot pierwszy", CalendarUtil.TOMORROW.getTime(), jandomanski);
        em.persist(item);
        itemIds[0] = item.getId();
        category.getItems().add(item);
        item.getCategories().add(category);
        for (int i = 1; i <= 3; i++) {
            Bid bid = new Bid(item, robertdomanski, new BigDecimal(9 + i));
            item.getBids().add(bid);
            em.persist(bid);
        }

        category = new Category("Kategoria druga");
        em.persist(category);
        categoryIds[1] = category.getId();

        item = new Item("Przedmiot drugi", CalendarUtil.TOMORROW.getTime(), jandomanski);
        em.persist(item);
        itemIds[1] = item.getId();
        category.getItems().add(item);
        item.getCategories().add(category);
        for (int i = 1; i <= 1; i++) {
            Bid bid = new Bid(item, janinadomanska, new BigDecimal(2 + i));
            item.getBids().add(bid);
            em.persist(bid);
        }

        item = new Item("Przedmiot trzeci", CalendarUtil.AFTER_TOMORROW.getTime(), janinadomanska);
        em.persist(item);
        itemIds[2] = item.getId();
        category.getItems().add(item);
        item.getCategories().add(category);

        category = new Category("Kategoria trzecia");
        em.persist(category);
        categoryIds[2] = category.getId();

        tx.commit();
        em.close();

        FetchTestData testData = new FetchTestData();
        testData.items = new TestData(itemIds);
        testData.users = new TestData(userIds);
        return testData;
    }

    @Test
    public void lazyEntityProxies() throws Exception {
        FetchTestData testData = storeTestData();

        UserTransaction tx = TM.getUserTransaction();
        try {
            tx.begin();
            EntityManager em = JPA.createEntityManager();

            Long ITEM_ID = testData.items.getFirstId();
            Long USER_ID = testData.users.getFirstId();

            {
                Item item = em.getReference(Item.class, ITEM_ID); // No SELECT

                //Wywołanie gettera identyfikatora (bez dostępu do pola!) nie powoduje inicjalizacji.
                assertEquals(item.getId(), ITEM_ID);

                //Klasa jest generowana w fazie działania aplikacji, a jej nazwa jest podobna do Item_ $$_ javassist_1
                assertNotEquals(item.getClass(), Item.class);

                assertEquals(
                   HibernateProxyHelper.getClassWithoutInitializingProxy(item),
                   Item.class
                );

                PersistenceUtil persistenceUtil = Persistence.getPersistenceUtil();
                assertFalse(persistenceUtil.isLoaded(item));
                assertFalse(persistenceUtil.isLoaded(item, "seller"));

                assertFalse(Hibernate.isInitialized(item));
                //Wykonanie instrukcji spowodowałoby inicjalizację obiektu Item!
                // assertFalse(Hibernate.isInitialized(item.getSeller()));

                Hibernate.initialize(item);
                // select * from ITEM where ID = ?

                //Zadbaj o to, aby domyślna opcja EAGER z @ManyToOne została przesłonięta opcją LAZY.
                assertFalse(Hibernate.isInitialized(item.getSeller()));

                Hibernate.initialize(item.getSeller());
                // select * from USERS where ID = ?
            }
            em.clear();
            {
                /* 
                   Egzemplarz encji <code>Item</code> jest ładowany w kontekście utrwalania, jego
                   <code>seller</code> nie jest zainicjowany. To jest obiekt proxy encji <code>User</code>.
                 */
                Item item = em.find(Item.class, ITEM_ID);
                // select * from ITEM where ID = ?

                /* 
                   Można ręcznie odłączyć dane z kontekstu utrwalania, albo zamknąć
                   kontekst utrwalania i odłączyć wszystko.
                 */
                em.detach(item);
                em.detach(item.getSeller());
                // em.close();

                /* 
                   Statyczna klasa pomocnicza <code>PersistenceUtil</code> działa bez kontekstu
                   utrwalania. W każdej chwili możemy sprawdzić, czy dane, do których chcemy uzyskać dostęp zostały załadowane.
                 */
                PersistenceUtil persistenceUtil = Persistence.getPersistenceUtil();
                assertTrue(persistenceUtil.isLoaded(item));
                assertFalse(persistenceUtil.isLoaded(item, "seller"));

                /* 
                   W stanie odłączonym możemy wywołać metodę gettera identyfikatora obiektu proxy User.
Jednak wywołanie dowolnej innej metody obiektu proxy, na przykład <code>getUsername()</code>, spowoduje zgłoszenie wyjątku <code>LazyInitializationException</code>.
                   Dane mogą zostać załadowane na żądanie tylko wtedy, kiedy kontekst utrwalania zarządza obiektem proxy. Nie mogą być załadowane w stanie odłączonym.

                 */
                assertEquals(item.getSeller().getId(), USER_ID);
                //Zgłasza wyjątek
                //assertNotNull(item.getSeller().getUsername());
            }
            em.clear();
            {
                //W tej procedurze nie ma żadnej instrukcji SQL SELECT, a tylko jedna instrukcja INSERT.
                Item item = em.getReference(Item.class, ITEM_ID);
                User user = em.getReference(User.class, USER_ID);

                Bid newBid = new Bid(new BigDecimal("99.00"));
                newBid.setItem(item);
                newBid.setBidder(user);

                em.persist(newBid);
                // insert into BID values (?, ? ,? , ...)

                em.flush();
                em.clear();
                assertEquals(em.find(Bid.class, newBid.getId()).getAmount().compareTo(new BigDecimal("99")), 0);
            }

            tx.commit();
            em.close();
        } finally {
            TM.rollback();
        }
    }

    @Test
    public void lazyCollections() throws Exception {
        FetchTestData testData = storeTestData();

        UserTransaction tx = TM.getUserTransaction();
        try {
            tx.begin();
            EntityManager em = JPA.createEntityManager();

            long ITEM_ID = testData.items.getFirstId();

            {
                Item item = em.find(Item.class, ITEM_ID);
                // select * from ITEM where ID = ?

                Set<Bid> bids = item.getBids(); // Kolekcja nie jest zainicjowana
                PersistenceUtil persistenceUtil = Persistence.getPersistenceUtil();
                assertFalse(persistenceUtil.isLoaded(item, "bids"));

                //To jest typ Set.
                assertTrue(Set.class.isAssignableFrom(bids.getClass())); 

                //To nie jest typ HashSet.
                assertNotEquals(bids.getClass(), HashSet.class);
                assertEquals(bids.getClass(), org.hibernate.collection.internal.PersistentSet.class);

                Bid firstBid = bids.iterator().next();
                // select * from BID where ITEM_ID = ?

                // Alternatywa: Hibernate.initialize(bids);
            }
            em.clear();
            {
                Item item = em.find(Item.class, ITEM_ID);
                // select * from ITEM where ID = ?

                assertEquals(item.getBids().size(), 3);
                // select count(b) from BID b where b.ITEM_ID = ?
            }

            tx.commit();
            em.close();
        } finally {
            TM.rollback();
        }
    }

}

