package org.jpwh.web.dao;

import javax.persistence.EntityManager;
import javax.persistence.TypedQuery;
import javax.persistence.criteria.*;
import javax.persistence.metamodel.Bindable;
import javax.persistence.metamodel.SingularAttribute;

public class SeekPage extends Page {

    /* 
		
		Technika przeszukiwania oprócz standardowego atrybutu sortującego 
		wymaga atrybutu stronicowania, który jest gwarantowanym unikatowym kluczem. 
		Może to być dowolny unikatowy atrybut modelu encji, ale zazwyczaj 
		jest to atrybut reprezentujący klucz główny.
		
     */
    protected SingularAttribute uniqueAttribute;

    /* 
		Zarówno dla atrybutu sortowania, jak i unikatowego klucza trzeba 
		zapamiętać wartości "z ostatniej strony". Można potem odtworzyć następną
		stronę poprzez wyszukanie tych wartości. Do tego celu można wykorzystać 
		dowolną wartość Comparable, zgodnie z wymaganiami API ograniczeń 
		w zapytaniach bazujących na kryteriach.		
		
     */
    protected Comparable lastValue;
    protected Comparable lastUniqueValue;

    public SeekPage(int size,
                    long totalRecords,
                    SingularAttribute defaultAttribute,
                    SortDirection defaultDirection,
                    SingularAttribute uniqueAttribute,
                    SingularAttribute... allowedAttributes) {
        super(size, totalRecords, defaultAttribute, defaultDirection, allowedAttributes);
        this.uniqueAttribute = uniqueAttribute;
    }

    public SingularAttribute getUniqueAttribute() {
        return uniqueAttribute;
    }

    public void setUniqueAttribute(SingularAttribute uniqueAttribute) {
        this.uniqueAttribute = uniqueAttribute;
    }

    public Comparable getLastValue() {
        return lastValue;
    }

    public void setLastValue(Comparable lastValue) {
        this.lastValue = lastValue;
    }

    public Comparable getLastUniqueValue() {
        return lastUniqueValue;
    }

    public void setLastUniqueValue(Comparable lastUniqueValue) {
        this.lastUniqueValue = lastUniqueValue;
    }

    public boolean isApplicableFor(Bindable bindable) {
        return super.isApplicableFor(bindable)
            && isAttributeDeclaredIn(getUniqueAttribute(), bindable);
    }

    public boolean isFirst() {
        return getLastValue() == null || getLastUniqueValue() == null;
    }

    @Override
    public <T> TypedQuery<T> createQuery(EntityManager em,
                                         CriteriaQuery<T> criteriaQuery,
                                         Path attributePath) {

        throwIfNotApplicableFor(attributePath);

        CriteriaBuilder cb = em.getCriteriaBuilder();

        /* 
            
			Zawsze należy posortować wyniki zarówno według atrybutu sortowania, 
			jak i według unikatowego klucza.
			
         */
        Path sortPath = attributePath.get(getSortAttribute());
        Path uniqueSortPath = attributePath.get(getUniqueAttribute());
        if (isSortedAscending()) {
            criteriaQuery.orderBy(cb.asc(sortPath), cb.asc(uniqueSortPath));
        } else {
            criteriaQuery.orderBy(cb.desc(sortPath), cb.desc(uniqueSortPath));
        }

        /* 
            			
			Dodanie do klauzuli where zapytania koniecznych dodatkowych 
			ograniczeń (tu ich nie pokazano), w celu wyszukiwania 
			poza ostatnio znanymi wartościami na docelowej stronie.
			
         */
        applySeekRestriction(em, criteriaQuery, attributePath);

        TypedQuery<T> query = em.createQuery(criteriaQuery);

        /* 
			Obcięcie wyników zgodnie z pożądanym rozmiarem strony.
         */
        if (getSize() != -1)
            query.setMaxResults(getSize());

        return query;
    }

    protected void applySeekRestriction(EntityManager em,
                                        CriteriaQuery criteriaQuery,
                                        Path attributePath) {
        // Nie trzeba nigdzie szukać, jeśli jesteśmy na pierwszej stronie
        if (isFirst())
            return;

        applySeekRestriction(
            em,
            criteriaQuery,
            attributePath,
            em.getCriteriaBuilder().literal(getLastValue()),
            em.getCriteriaBuilder().literal(getLastUniqueValue())
        );
    }

    protected void applySeekRestriction(EntityManager em,
                                        AbstractQuery criteriaQuery,
                                        Path attributePath,
                                        Expression lastValueExpression,
                                        Expression lastUniqueValueExpression) {
        CriteriaBuilder cb = em.getCriteriaBuilder();
        criteriaQuery.where(
            cb.and(
                (isSortedAscending()
                    ? cb.greaterThanOrEqualTo(
                    attributePath.get(getSortAttribute()),
                    lastValueExpression)
                    : cb.lessThanOrEqualTo(
                    attributePath.get(getSortAttribute()),
                    lastValueExpression)
                ),
                cb.or(
                    cb.notEqual(
                        attributePath.get(getSortAttribute()),
                        lastValueExpression
                    ),
                    (isSortedAscending()
                        ? cb.greaterThan(
                        attributePath.get(getUniqueAttribute()),
                        lastUniqueValueExpression)
                        : cb.lessThan(
                        attributePath.get(getUniqueAttribute()),
                        lastUniqueValueExpression)
                    )
                )
            )
        );
    }
}
