package org.jpwh.web.dao;

import javax.persistence.EntityManager;
import javax.persistence.EntityManagerFactory;
import javax.persistence.PersistenceUnit;

/* 
    Ta adnotacja CDI jest deklaracją mówiącą o tym, że w całej aplikacji jest potrzebny tylko jeden producent:
    zawsze będzie tylko jeden egzemplarz 
    <code>EntityManagerProducer</code>.
 */
@javax.enterprise.context.ApplicationScoped
public class EntityManagerProducer {

    /* 
        Środowisko Java EE runtime zwróci jednostkę utrwalania 
        skonfigurowanę w pliku <code>persistence.xml</code>, który jednocześnie jest
        komponentem zasięgu aplikacji. (jeśli korzystasz z samodzielnej instalacji CDI poza środowiskiem
        Java EE, to możesz zamiast tego użyć statycznego 
        kodu inicjującego <code>Persistence.createEntityManagerFactory()</code>.)

     */
    @PersistenceUnit
    private EntityManagerFactory entityManagerFactory;

    /* 
        Zawsze, gdy jest potrzebny obiekt <code>EntityManager</code>, wywoływana jest metoda <code>create()</code>.
        Kontener wykorzystuje wielokrotnie ten sam obiekt <code>EntityManager</code>
        podczas obsługi żądania przez serwer. (jeśli nie dodamy adnotacji
        <code>@RequestScoped</code> do metody, to obiekt
        <code>EntityManager</code> będzie miał zasięg aplikacji tak, jak klasa producenta!)
     */
    @javax.enterprise.inject.Produces
    @javax.enterprise.context.RequestScoped
    public EntityManager create() {
        return entityManagerFactory.createEntityManager();
    }

    /* 
        Po zakończeniu obsługi żądania i zniszczeniu kontekstu żądania, kontener CDI
        wywoła tę metodę, aby pozbyć się egzemplarza obiektu <code>EntityManager</code>.
		Ponieważ utworzyliśmy kontekst utrwalania zarządzany przez aplikację
        (patrz <a href="#PersistenceContext"/>), to nasztm zadaniem jest jego zamknięcie.
     */
    public void dispose(@javax.enterprise.inject.Disposes
                        EntityManager entityManager) {
        if (entityManager.isOpen())
            entityManager.close();
    }
}

