package org.jpwh.env;

import bitronix.tm.TransactionManagerServices;
import bitronix.tm.resource.jdbc.PoolingDataSource;

import javax.naming.Context;
import javax.naming.InitialContext;
import javax.sql.DataSource;
import javax.transaction.Status;
import javax.transaction.UserTransaction;
import java.util.logging.Logger;

/**
 * Dostarcza pulę połączeń bazy danych z menedżerem transakcji Bitronix JTA
 * (http://docs.codehaus.org/display/BTM/Home).
 * <p>
 * Hibernate odszuka źródło danych i <code>UserTransaction</code> za pośrednictwem 
 * JNDI. Dlatego właśnie potrzebny jest również plik <code>jndi.properties</code>. Minimalny kontekst
 * JNDI jest dołączony i uruchamiany przez Bitronix.
 * </p>
 */
public class TransactionManagerSetup {

    public static final String DATASOURCE_NAME = "myDS";

    private static final Logger logger =
        Logger.getLogger(TransactionManagerSetup.class.getName());

    protected final Context context = new InitialContext();
    protected final PoolingDataSource datasource;
    public final DatabaseProduct databaseProduct;

    public TransactionManagerSetup(DatabaseProduct databaseProduct) throws Exception {
        this(databaseProduct, null);
    }

    public TransactionManagerSetup(DatabaseProduct databaseProduct,
                                   String connectionURL) throws Exception {

        logger.fine("Uruchamianie puli połączeń z bazą danych");

        logger.fine("Ustawienie stabilnego, unikatowego identyfikatora w celu przywrócenia transakcji");
        TransactionManagerServices.getConfiguration().setServerId("myServer1234");

        logger.fine("Wyłączenie dowiązania JMX menedżera w testach jednostkowych");
        TransactionManagerServices.getConfiguration().setDisableJmx(true);

        logger.fine("Wyłączenie logowania transakcji dla testów jednostkowych");
        TransactionManagerServices.getConfiguration().setJournal("null");

        logger.fine("Wyłączenie ostrzeżeń w sytuacji, gdy dostęp do bazy danych odbywa się poza transakcją");
        TransactionManagerServices.getConfiguration().setWarnAboutZeroResourceTransaction(false);

        logger.fine("Tworzenie puli połączeń");
        datasource = new PoolingDataSource();
        datasource.setUniqueName(DATASOURCE_NAME);
        datasource.setMinPoolSize(1);
        datasource.setMaxPoolSize(5);
        datasource.setPreparedStatementCacheSize(10);

        // Testy blokowania/wersjonowania zakładają izolację transakcji READ COMMITTED.
        // To nie jest ustawienie domyślne dla bazy MySQL InnoDB, dlatego ustawiamy je
        // tutaj jawnie.
        datasource.setIsolationLevel("READ_COMMITTED");

        // Generator schematu SQL frameworka Hibernate wywołuje metodę connection.setAutoCommit(true).
        // Używamy trybu autozatwierdzania w przypadku, gdy obiekt EntityManager jest w trybie uśpionym
        // i nie jest powiązany z transakcją.
        datasource.setAllowLocalTransactions(true);

        logger.info("Konfigurowanie połączenia z bazą danych: " + databaseProduct);
        this.databaseProduct = databaseProduct;
        databaseProduct.configuration.configure(datasource, connectionURL);

        logger.fine("Inicjowanie transakcji i zarządzanie zasobami");
        datasource.init();
    }

    public Context getNamingContext() {
        return context;
    }

    public UserTransaction getUserTransaction() {
        try {
            return (UserTransaction) getNamingContext()
                .lookup("java:comp/UserTransaction");
        } catch (Exception ex) {
            throw new RuntimeException(ex);
        }
    }

    public DataSource getDataSource() {
        try {
            return (DataSource) getNamingContext().lookup(DATASOURCE_NAME);
        } catch (Exception ex) {
            throw new RuntimeException(ex);
        }
    }

    public void rollback() {
        UserTransaction tx = getUserTransaction();
        try {
            if (tx.getStatus() == Status.STATUS_ACTIVE ||
                tx.getStatus() == Status.STATUS_MARKED_ROLLBACK)
                tx.rollback();
        } catch (Exception ex) {
            System.err.println("Cofnięcie transakcji nie powiodło się. Poniżej ślad!");
            ex.printStackTrace(System.err);
        }
    }

    public void stop() throws Exception {
        logger.fine("Zatrzymanie puli połączeń z bazą danych");
        datasource.close();
        TransactionManagerServices.getTransactionManager().shutdown();
    }

}

