package org.jpwh.stateful;

import org.jpwh.dao.BidDAO;
import org.jpwh.dao.ItemDAO;
import org.jpwh.model.Bid;
import org.jpwh.model.InvalidBidException;
import org.jpwh.model.Item;

import javax.inject.Inject;
import javax.persistence.EntityManager;
import javax.persistence.EntityNotFoundException;
import javax.persistence.PersistenceContext;
import java.math.BigDecimal;

import static javax.persistence.LockModeType.OPTIMISTIC_FORCE_INCREMENT;
import static javax.persistence.PersistenceContextType.EXTENDED;
import static javax.persistence.SynchronizationType.UNSYNCHRONIZED;

@javax.ejb.Stateful(passivationCapable = false)
@javax.ejb.StatefulTimeout(10) // Minuty
@javax.ejb.Local(ItemService.class)
@javax.ejb.Remote(RemoteItemService.class)
public class ItemServiceImpl implements ItemService {

    @PersistenceContext(type = EXTENDED, synchronization = UNSYNCHRONIZED)
    protected EntityManager em;

    @Inject
    protected ItemDAO itemDAO;

    @Inject
    protected BidDAO bidDAO;

    // stan konwersacji po stronie serwera
    protected Item item;
    protected boolean isItemLockRequired;

    @Override
    public void startConversation(Long itemId) {
        item = itemDAO.findById(itemId);
        if (item == null)
            throw new EntityNotFoundException(
                "Nie znaleziono przedmiotu o identyfikatorze: " + itemId
            );
    }

    @Override
    public void setItemName(String newName) {
        item.setName(newName);
        // Kontekst utrwalania NIE zostanie zsynchronizowany w momencie zatwierdzania
        // transakcji. Pozostanie niezsynchronizowany!
    }

    @Override
    public void placeBid(BigDecimal amount) throws InvalidBidException {
        Bid bid = new Bid(amount, item);

        // Sprawdzenie, czy zostały spełnione reguły biznesowe
        if (!bid.getItem().isValidBid(bid))
            throw new InvalidBidException("Kwota oferty jest zbyt niska!");

        // W momencie wykonywania synchronizacji MUSI być sprawdzona wersja przedmiotu
        isItemLockRequired = true;

        item.getBids().add(bid);
        bidDAO.makePersistent(bid);
        // Kontekst utrwalania NIE zostanie zsynchronizowany w momencie zatwierdzania
        // transakcji. Pozostanie niezsynchronizowany!
    }

    @Override
    @javax.ejb.Remove // Komponent bean zostanie usunięty po zakończeniu działania tej metody
    public void commitConversation() {
        em.joinTransaction();
        // Kontekst utrwalania jest scalony z bieżącą transakcją i będzie
        // zsynchronizowany kiedy ta metoda zwróci sterowanie. To spowoduje zapisanie zmian.

        // Teraz menedżer encji EM jest scalony z transakcją, dlatego możemy go zablokować
        if (isItemLockRequired)
            em.lock(item, OPTIMISTIC_FORCE_INCREMENT);
    }
}

