package synch;

import java.util.*;
import java.util.concurrent.locks.*;

/**
 * Bank z kilkoma kontami, ktry uywa blokad do serializacji dostpu.
 */
public class Bank
{
   private final double[] accounts;
   private final Lock bankLock = new ReentrantLock();
   private final Condition sufficientFunds = bankLock.newCondition();

    /**
    * Tworzy bank
    * @param n liczba kont
    * @param initialBalance saldo pocztkowe na kadym koncie
    */
   public Bank(int n, double initialBalance)
   {
      accounts = new double[n];
      Arrays.fill(accounts, initialBalance);
   }

   /**
    * Przelewa pienidze pomidzy kontami.
    * @param from konto, z ktrego ma nastpi przelew
    * @param to konto, na ktre maj zosta przelane rodki
    * @param amount kwota do przelania
    */
   public void transfer(int from, int to, double amount) throws InterruptedException
   {
      bankLock.lock();
      try
      {
         while (accounts[from] < amount)
            sufficientFunds.await();
         System.out.print(Thread.currentThread());
         accounts[from] -= amount;
         System.out.printf(" %10.2f z %d na %d", amount, from, to);
         accounts[to] += amount;
         System.out.printf(" Saldo oglne: %10.2f%n", getTotalBalance());
         sufficientFunds.signalAll();
      }
      finally
      {
         bankLock.unlock();
      }
   }

   /**
    * Zwraca sum sald wszystkich kont.
    * @return saldo oglne
    */
   public double getTotalBalance()
   {
      bankLock.lock();
      try
      {
         double sum = 0;

         for (double a : accounts)
            sum += a;

         return sum;
      }
      finally
      {
         bankLock.unlock();
      }
   }

   /**
    * Zwraca liczb kont w banku.
    * @return liczba kont
    */
   public int size()
   {
      return accounts.length;
   }
}