package com.corejsf;

import java.io.IOException;
import java.util.Map;
import java.util.logging.Level;
import java.util.logging.Logger;
import javax.faces.component.UIComponent;
import javax.faces.context.ExternalContext;
import javax.faces.context.FacesContext;
import javax.faces.context.ResponseWriter;
import javax.faces.event.ActionEvent;
import javax.faces.model.SelectItem;
import javax.faces.render.Renderer;
import javax.servlet.ServletContext;
import javax.servlet.ServletException;
import javax.servlet.ServletRequest;
import javax.servlet.ServletResponse;

// Klasa odpowiedzialna za wizualizacj komponentu UITabbedPane

public class TabbedPaneRenderer extends Renderer {
   private static Logger logger = Logger.getLogger("com.corejsf.util");

   // Metoda getRendersChildren() domylnie zwraca warto false, zatem metoda encodeChildren() nie zostanie
   // wywoana, dopki nie nadpiszemy metody getRendersChildren() wersj zwracajc warto true.

   public boolean getRendersChildren() {
      return true;
   }

   // Metoda decode uzyskuje warto parametru dania, ktrego nazwa odpowiada identyfikatorowi
   // klienta komponentu panelu podzielonego na zakadki. Odpowiedni parametr dania jest kodowany
   // przez metod encodeHiddenField (wywoywan przez metod encodeEnd) w formie pola ukrytego.
   // Warto tego parametru jest ustawiana przez skrypt jzyka JavaScript wygenerowany przez metod
   // encodeTab i reprezentuje nazw facety lub strony JSP.

   // Metoda decode wykorzystuje warto tego parametru dania do ustawienia
   // atrybutu treci komponentu panelu podzielonego na zakadki.
   // I wreszcie metoda decode() umieszcza zdarzenie akcji w kolejce, z ktrej zostanie
   // przekazane metodom nasuchujcym w fazie wywoania aplikacji cyklu ycia JSF.
   // Metody nasuchujce akcji mona definiowa za porednictwem atrybutu
   // actionListener znacznika <corejsf:tabbedpane> lub znacznikw <f:actionListener>
   // w ciele znacznika <corejsf:tabbedpane>.

   public void decode(FacesContext context, UIComponent component) {
      Map<String, String> requestParams 
         = context.getExternalContext().getRequestParameterMap();
      String clientId = component.getClientId(context);

      String content = (String) (requestParams.get(clientId));
      if (content != null && !content.equals("")) {
         UITabbedPane tabbedPane = (UITabbedPane) component;
         tabbedPane.setContent(content);
      }
      
      component.queueEvent(new ActionEvent(component));
   }

   // Metoda encodeBegin zapisuje pocztkowy element <table> jzyka HTML
   // wraz z klas CSS wskazywan za porednictwem atrybutu styleClass
   // znacznika <corejsf:tabbedpane> (jeli taki atrybut w ogle zdefiniowano).

   public void encodeBegin(FacesContext context, UIComponent component)
         throws java.io.IOException {
      ResponseWriter writer = context.getResponseWriter();
      writer.startElement("table", component);

      String styleClass = (String) component.getAttributes().get("styleClass");
      if (styleClass != null)
         writer.writeAttribute("class", styleClass, null);

      writer.write("\n"); // taki krok poprawia czytelno generowanego kodu HTML-a
   }

  // Metoda encodeChildren() jest wywoywana przez implementacj JSF po metodzie encodeBegin().
  // Elementy potomne wzgldem komponentu <corejsf:tabbedpane> maj posta komponentw UISelectItem
  // ustawionych za porednictwem jednego lub wielu znacznikw <f:selectItem> bd pojedynczego znacznika
  // <f:selectItems> w ciele znacznika <corejsf:tabbedpane>.

   public void encodeChildren(FacesContext context, UIComponent component)
         throws java.io.IOException {
      // Metoda encodeChildren jest wywoywana, nawet jeli komponent tabbedpane
      // nie obejmuje komponentw potomnych.
      if (component.getChildCount() == 0) {
         return;
      }

      ResponseWriter writer = context.getResponseWriter();
      writer.startElement("thead", component);
      writer.startElement("tr", component);
      writer.startElement("th", component);

      writer.startElement("table", component);
      writer.startElement("tbody", component);
      writer.startElement("tr", component);

      for (SelectItem item : com.corejsf.util.Renderers.getSelectItems(component))
         encodeTab(context, writer, item, component);
      
      writer.endElement("tr");
      writer.endElement("tbody");
      writer.endElement("table");

      writer.endElement("th");
      writer.endElement("tr");
      writer.endElement("thead");
      writer.write("\n"); // taki krok poprawia czytelno generowanego kodu HTML-a
   }

   // Metoda encodeEnd() jest wywoywana przez implementacj JSF po metodzie encodeChildren().
   // Metoda encodeEnd() zapisuje ciao tabeli i koduje zawarto komponentu tabbedpane
   // w ramach pojedynczego wiersza tej tabeli.

   // Tre panelu podzielonego na zakadki okrelamy albo w formie adresu URL
   // wskazujcego na stron JSP, albo w formie nazwy facety. Metoda encodeEnd()
   // sprawdza, czy wskazano facet; jeli tak, koduje j; jeli nie, docza stron JSP.

   public void encodeEnd(FacesContext context, UIComponent component)
         throws java.io.IOException {
      ResponseWriter writer = context.getResponseWriter();
      UITabbedPane tabbedPane = (UITabbedPane) component;
      String content = tabbedPane.getContent();

      writer.startElement("tbody", component);
      writer.startElement("tr", component);
      writer.startElement("td", component);

      if (content != null) {
         UIComponent facet = component.getFacet(content);
         if (facet != null) {
            if (facet.isRendered()) {
               facet.encodeBegin(context);
               if (facet.getRendersChildren())
                  facet.encodeChildren(context);
               facet.encodeEnd(context);
            }
         } else
            includePage(context, component);
      }

      writer.endElement("td");
      writer.endElement("tr");
      writer.endElement("tbody");

      // Zamyka elementy kolumny, wiersza i tabeli.
      writer.endElement("table");

      encodeHiddenField(context, writer, component);
   }

   // Metoda encodeHiddenField jest wywoywana na kocu metody encodeEnd(). Znaczenie
   // pola ukrytego i jego wartoci wyjaniono przy okazji omawiania metody decode.

   private void encodeHiddenField(FacesContext context, ResponseWriter writer,
         UIComponent component) throws java.io.IOException {
      // Zapisuje pole ukryte, ktrego nazwa odpowiada identyfikatorowi klienta.
      writer.startElement("input", component);
      writer.writeAttribute("type", "hidden", null);
      writer.writeAttribute("name", component.getClientId(context), null);
      writer.endElement("input");
   }

   // Metoda encodeTab (wywoywana z poziomu metody encodeChildren) koduje element kotwicy jzyka
   // HTML z atrybutem onclick, ktry wymusza ustawianie wartoci pola ukrytego zakodowanego przez
   // metod encodeHiddenField i wysanie na serwer caego formularza. Wicej informacji na temat
   // samego pola ukrytego mona znale w komentarzu powiconym metodzie decode. Metoda encodeTab
   // dodatkowo zapisuje atrybut class dla poszczeglnych zakadek; warto tego atrybutu wskazuje albo
   // na atrybut tabClass (w przypadku niezaznaczonych zakadek), albo na atrybut selectedTabClass
   // (w przypadku zakadki zaznaczonej).

   private void encodeTab(FacesContext context, ResponseWriter writer,
         SelectItem item, UIComponent component) throws java.io.IOException {
      String tabText = getLocalizedTabText(component, item.getLabel());
      String content = (String) item.getValue();

      writer.startElement("td", component);
      writer.startElement("a", component);
      writer.writeAttribute("href", "#", "href");

      String clientId = component.getClientId(context);
      String formId = com.corejsf.util.Renderers.getFormId(context, component);

      writer.writeAttribute("onclick",
      // Zapisuje warto pola ukrytego, ktrego nazwa odpowiada identyfikatorowi klienta.

            "document.forms['" + formId + "']['" + clientId + "'].value='"
                  + content + "'; " +

                  // Wysya formularz, do ktrego naley komponent panelu.
                  "document.forms['" + formId + "'].submit(); ", null);

      UITabbedPane tabbedPane = (UITabbedPane) component;
      String selectedContent = tabbedPane.getContent(); 

      String tabClass = null;
      if (content.equals(selectedContent))
         tabClass = (String) component.getAttributes().get("selectedTabClass");
      else
         tabClass = (String) component.getAttributes().get("tabClass");

      if (tabClass != null)
         writer.writeAttribute("class", tabClass, null);

      writer.write(tabText);

      writer.endElement("a");
      writer.endElement("td");
      writer.write("\n"); // taki krok poprawia czytelno generowanego kodu HTML-a
   }

   // Tekst waciwy dla zakadek naszego komponentu panelu moemy zdefiniowa albo z wykorzystaniem
   // kluczy pakietu komunikatw, albo bezporednio w formie tekstu wywietlanego w zakadce. Na podstawie
   // tego tekstu metoda getLocalizedTabText prbuje odnale odpowiedni warto w pakiecie komunikatw
   // wskazanym za porednictwem atrybutu resourceBundle znacznika <corejsf:tabbedpane>. Jeli prba
   // odnalezienia tej wartoci zakoczy si niepowodzeniem, metoda getLocalizedTabText zwraca acuch,
   // ktry przekazano na jej wejciu.

   private String getLocalizedTabText(UIComponent tabbedPane, String key) {
      String bundle = (String) tabbedPane.getAttributes().get("resourceBundle");
      String localizedText = null;

      if (bundle != null) {
         localizedText = com.corejsf.util.Messages.getString(bundle, key, null);
      }
      if (localizedText == null)
         localizedText = key;
      // Jeli parametr key nie reprezentuje istniejcego klucza w pakiecie komunikatw,
      // zwracamy acuch wejciowy w niezmienionej formie.
      return localizedText;
   }

   // Metoda includePage wykorzystuje mechanizm rozdziau da serwletu do
   // doczania strony waciwej dla wybranej zakadki.

   private void includePage(FacesContext fc, UIComponent component) {
	  ExternalContext ec = fc.getExternalContext();
      ServletContext sc = (ServletContext) ec.getContext();
      UITabbedPane tabbedPane = (UITabbedPane) component;
      String content = tabbedPane.getContent();
      
      ServletRequest request = (ServletRequest) ec.getRequest();
      ServletResponse response = (ServletResponse) ec.getResponse();
      try {
         sc.getRequestDispatcher(content).include(request, response);
      } catch (ServletException ex) {
         logger.log(Level.WARNING, "Couldn't load page: " + content, ex);
      } catch (IOException ex) {
         logger.log(Level.WARNING, "Couldn't load page: " + content, ex);
      }
   }
}
