package com.corejsf;

import java.io.IOException;
import java.util.Map;
import java.util.logging.Level;
import java.util.logging.Logger;
import javax.faces.component.UIComponent;
import javax.faces.context.ExternalContext;
import javax.faces.context.FacesContext;
import javax.faces.context.ResponseWriter;
import javax.faces.event.ActionEvent;
import javax.faces.model.SelectItem;
import javax.faces.render.FacesRenderer;
import javax.faces.render.Renderer;
import javax.servlet.ServletContext;
import javax.servlet.ServletException;
import javax.servlet.ServletRequest;
import javax.servlet.ServletResponse;

// Klasa odpowiedzialna za wizualizacj komponentu UITabbedPane

@FacesRenderer(componentFamily="javax.faces.Command", 
   rendererType="com.corejsf.TabbedPane")
public class TabbedPaneRenderer extends Renderer {
   private static Logger logger = Logger.getLogger("com.corejsf.util");

   // Metoda getRendersChildren() domylnie zwraca warto false, zatem metoda encodeChildren() nie zostanie
   // wywoana, chyba e przykryjemy metod getRendersChildren() wersj zwracajc warto true.

   public boolean getRendersChildren() {
      return true;
   }

   // Metoda decode odczytuje warto parametru dania, ktrego nazwa odpowiada identyfikatorowi
   // komponentu tabbedpane stosowanemu po stronie klienta. Wspomniany parametr dania jest
   // kodowany w formie pola ukrytego przez metod encodeHiddenField (wywoywan przez metod
   // encodeEnd). Warto tego parametru jest ustawiana przez kod JavaScriptu wygenerowany
   // przez metod encodeTab. W roli tej wartoci stosuje si nazw facety lub strony JSF.

   // Metoda decode uywa wartoci parametru dania do ustawienia atrybutu content
   // komponentu tabbedpane.
   // I wreszcie metoda decode() kolejkuje zdarzenie akcji kierowane nastpnie (w fazie
   // wywoywania aplikacji cyklu ycia JSF) do zarejestrowanych mechanizmw
   // nasuchujcych. Wspomniane mechanizmy mona wskazywa albo za pomoc
   // atrybutu actionListener znacznika <corejsf:tabbedpane>, albo za pomoc znacznikw
   // <f:actionListener> w ciele znacznika <corejsf:tabbedpane>.

   public void decode(FacesContext context, UIComponent component) {
      Map<String, String> requestParams 
         = context.getExternalContext().getRequestParameterMap();
      String clientId = component.getClientId(context);

      String content = (String) (requestParams.get(clientId));
      if (content != null && !content.equals("")) {
         UITabbedPane tabbedPane = (UITabbedPane) component;
         tabbedPane.setContent(content);
      }
      
      component.queueEvent(new ActionEvent(component));
   }

   // Metoda encodeBegin zapisuje pocztkowy element <table> jzyka HTML
   // z klas CSS wskazan przez atrybut styleClass znacznika <corejsf:tabbedpane>
   // (jeli ten znacznik zdefiniowano).

   public void encodeBegin(FacesContext context, UIComponent component)
         throws java.io.IOException {
      ResponseWriter writer = context.getResponseWriter();
      writer.startElement("table", component);

      String styleClass = (String) component.getAttributes().get("styleClass");
      if (styleClass != null)
         writer.writeAttribute("class", styleClass, null);

      writer.write("\n"); // w ten sposb poprawiamy czytelno generowanego kodu HTML-a
   }

   // Metoda encodeChildren() jest wywoywana przez implementacj JSF po metodzie encodeBegin().
   // Komponenty potomne komponentu <corejsf:tabbedpane> maj posta komponentw typu UISelectItem
   // ustawionych za pomoc jednego lub wielu znacznikw <f:selectItem> bd pojedynczego
   // znacznika <f:selectItems> w ciele znacznika <corejsf:tabbedpane>.

   public void encodeChildren(FacesContext context, UIComponent component)
         throws java.io.IOException {
      // Ta metoda jest wywoywana, nawet jeli komponent tabbedpane nie zawiera adnych
      // komponentw potomnych.
      if (component.getChildCount() == 0) {
         return;
      }

      ResponseWriter writer = context.getResponseWriter();
      writer.startElement("thead", component);
      writer.startElement("tr", component);
      writer.startElement("th", component);

      writer.startElement("table", component);
      writer.startElement("tbody", component);
      writer.startElement("tr", component);

      for (SelectItem item : com.corejsf.util.Renderers.getSelectItems(component))
         encodeTab(context, writer, item, component);
      
      writer.endElement("tr");
      writer.endElement("tbody");
      writer.endElement("table");

      writer.endElement("th");
      writer.endElement("tr");
      writer.endElement("thead");
      writer.write("\n"); // poprawia czytelno generowanego kodu HTML
   }

   // Metoda encodeEnd() jest wywoywana przez implementacj JSF po metodzie encodeChildren().
   // Metoda encodeEnd() zapisuje ciao tabeli i koduje tre panelu podzielonego na zakadki
   // w jednym wierszu.

   // Tre panelu podzielonego na zakadki mona okreli albo w formie adresu URL strony JSF,
   // albo w formie nazwy facety, zatem metoda encodeEnd() sprawdza, czy ma do czynienia z facet;
   // jeli tak, metoda koduje t facet; w przeciwnym razie metoda docza odpowiedni stron JSF.

   public void encodeEnd(FacesContext context, UIComponent component)
         throws java.io.IOException {
      ResponseWriter writer = context.getResponseWriter();
      UITabbedPane tabbedPane = (UITabbedPane) component;
      String content = tabbedPane.getContent();

      writer.startElement("tbody", component);
      writer.startElement("tr", component);
      writer.startElement("td", component);

      if (content != null) {
         UIComponent facet = component.getFacet(content);
         if (facet != null) {
            if (facet.isRendered()) {
               facet.encodeBegin(context);
               if (facet.getRendersChildren())
                  facet.encodeChildren(context);
               facet.encodeEnd(context);
            }
         } else
            includePage(context, component);
      }

      writer.endElement("td");
      writer.endElement("tr");
      writer.endElement("tbody");

      // Zamyka elementy kolumny, wiersza i tabeli
      writer.endElement("table");

      encodeHiddenField(context, writer, component);
   }

   // Metoda encodeHiddenField jest wywoywana na kocu metody encodeEnd().
   // Wyjanienie znaczenia pola ukrytego i jego wartoci mona znale w metodzie decode.

   private void encodeHiddenField(FacesContext context, ResponseWriter writer,
         UIComponent component) throws java.io.IOException {
      // Zapisuje pole ukryte, ktrego nazwa odpowiada identyfikatorowi komponentu tabbedpane po stronie klienta.
      writer.startElement("input", component);
      writer.writeAttribute("type", "hidden", null);
      writer.writeAttribute("name", component.getClientId(context), null);
      writer.endElement("input");
   }

   // Metoda encodeTab (wywoywana przez metod encodeChildren) koduje element kotwicy
   // jzyka HTML z atrybutem onclick ustawiajcym warto pola ukrytego (zakodowanego przez
   // metod encodeHiddenField) i wysya formularz zawierajcy komponent tabbedpane. Wicej
   // informacji na temat tego pola ukrytego mona znale w metodzie decode.
   // Metoda encodeTab dodatkowo zapisuje atrybut klasy dla kadej zakadki odpowiadajcy albo
   // atrybutowi tabClass (w przypadku niezaznaczonych zakadek), albo atrybutowi
   // selectedTabClass (w przypadku zaznaczonej zakadki).


   private void encodeTab(FacesContext context, ResponseWriter writer,
         SelectItem item, UIComponent component) throws java.io.IOException {
      String tabText = getLocalizedTabText(component, item.getLabel());
      String content = (String) item.getValue();

      writer.startElement("td", component);
      writer.startElement("a", component);
      writer.writeAttribute("href", "#", "href");

      String clientId = component.getClientId(context);
      String formId = com.corejsf.util.Renderers.getFormId(context, component);

      writer.writeAttribute("onclick",
      // zapisuje warto pola ukrytego, ktrego nazwa odpowiada identyfikatorowi komponentu tabbedpane klienta

            "document.forms['" + formId + "']['" + clientId + "'].value='"
                  + content + "'; " +

                  // wysya formularz zawierajcy komponent
                  "document.forms['" + formId + "'].submit(); ", null);

      UITabbedPane tabbedPane = (UITabbedPane) component;
      String selectedContent = tabbedPane.getContent(); 

      String tabClass = null;
      if (content.equals(selectedContent))
         tabClass = (String) component.getAttributes().get("selectedTabClass");
      else
         tabClass = (String) component.getAttributes().get("tabClass");

      if (tabClass != null)
         writer.writeAttribute("class", tabClass, null);

      writer.write(tabText);

      writer.endElement("a");
      writer.endElement("td");
      writer.write("\n"); // to make generated HTML easier to read
   }

   // Tekst zakadek komponentu tabbedpane mona wskaza w formie
   // klucza w pakiecie zasobw lub bezporednio w formie tekstu wywietlanego
   // w zakadce. Na podstawie tego tekstu metoda getLocalizedTabText prbuje
   // uzyska warto z pakietu komunikatw wskazanego przy uyciu atrybutu
   // resourceBundle znacznika <corejsf:tabbedpane>. W razie braku wartoci
   // metoda getLocalizedTabText zwraca acuch, ktry przekazano na jej wejciu.

   private String getLocalizedTabText(UIComponent tabbedPane, String key) {
      String bundle = (String) tabbedPane.getAttributes().get("resourceBundle");
      String localizedText = null;

      if (bundle != null) {
         localizedText = com.corejsf.util.Messages.getString(bundle, key, null);
      }
      if (localizedText == null)
         localizedText = key;
      // Parametr key w rzeczywistoci nie reprezentuje klucza w pakiecie komunikatw,
      // zatem metoda zwraca acuch w niezmienionej formie.
      return localizedText;
   }

   // Metoda includePage uywa mechanizmu kierowania da do serwletw, aby doczy
   // stron waciw wybranej zakadce.

   private void includePage(FacesContext fc, UIComponent component) {
      ExternalContext ec = fc.getExternalContext();
      ServletContext sc = (ServletContext) ec.getContext();
      UITabbedPane tabbedPane = (UITabbedPane) component;
      String content = tabbedPane.getContent();
      
      ServletRequest request = (ServletRequest) ec.getRequest();
      ServletResponse response = (ServletResponse) ec.getResponse();
      try {
         sc.getRequestDispatcher(content).include(request, response);
      } catch (ServletException ex) {
         logger.log(Level.WARNING, "Nie mona zaadowa strony: " + content, ex);
      } catch (IOException ex) {
         logger.log(Level.WARNING, "Nie mona zaadowa stronye: " + content, ex);
      }
   }
}
