import java.util.*;

public final class PhoneNumber implements Cloneable, Comparable<PhoneNumber> {
    private final short areaCode;
    private final short prefix;
    private final short lineNumber;

    public PhoneNumber(int areaCode, int prefix,
                       int lineNumber) {
        rangeCheck(areaCode,   999, "nr kierunkowy");
        rangeCheck(prefix,   999, "prefiks");
        rangeCheck(lineNumber, 9999, "numer linii");
        this.areaCode  = (short) areaCode;
        this.prefix  = (short) prefix;
        this.lineNumber = (short) lineNumber;
    }

    private static void rangeCheck(int arg, int max,
                                   String name) {
        if (arg < 0 || arg > max)
           throw new IllegalArgumentException(name +": " + arg);
    }

    @Override public boolean equals(Object o) {
        if (o == this)
            return true;
        if (!(o instanceof PhoneNumber))
            return false;
        PhoneNumber pn = (PhoneNumber)o;
        return pn.lineNumber == lineNumber
            && pn.prefix  == prefix
            && pn.areaCode  == areaCode;
    }

    @Override public int hashCode() {
        int result = 17;
        result = 31 * result + areaCode;
        result = 31 * result + prefix;
        result = 31 * result + lineNumber;
        return result;
    }

    /**
     * Zwraca cig reprezentujcy ten numer telefonu.
     * Cig skada si z czternastu znakw w formacie
     * "(XXX) YYY-ZZZZ", gdzie XXX jest numerem kierunkowym, YYY to
     * centrala, a ZZZZ rozszerzenie (kada litera reprezentuje
     * jedn cyfr).
     *
     * Jeeli dowolna z tych trzech czci jest zbyt maa,
     * aby wypeni swoje pole, jest ono uzupeniane z lewej strony zerami.
     * Na przykad, jeeli numerem linii jest 123, ostatnie
     * cztery znaki cigu bd miay posta "0123".
     *
     * Za zamykajcym nawiasem znajduje si spacja, oddzielajca
     * numer kierunkowy od prefiksu.
     */
    @Override public String toString() {
        return String.format("(%03d) %03d-%04d",
                             areaCode, prefix, lineNumber);
    }

    @Override public PhoneNumber clone() {
        try {
            return (PhoneNumber) super.clone();
        } catch(CloneNotSupportedException e) {
            throw new AssertionError();
        }
    }

    public int compareTo(PhoneNumber pn) {
        int areaCodeDiff = areaCode - pn.areaCode;
        if (areaCodeDiff != 0)
            return areaCodeDiff;

        int prefixDiff = prefix - pn.prefix;
        if (prefixDiff != 0)
            return prefixDiff;

        return lineNumber - pn.lineNumber;
    }

    public static void main(String[] args) {
        NavigableSet<PhoneNumber> s = new TreeSet<PhoneNumber>();
        for (int i = 0; i < 10; i++)
            s.add(randomPhoneNumber());
        System.out.println(s);
    }

    private static final Random rnd = new Random();

    private static PhoneNumber randomPhoneNumber() {
        return new PhoneNumber((short) rnd.nextInt(1000),
                               (short) rnd.nextInt(1000),
                               (short) rnd.nextInt(10000));
    }
}
