package com.wakaleo.jpt.taxcalculator.impl;

import static org.junit.Assert.*;
import static org.hamcrest.Matchers.*;

import java.util.ArrayList;
import java.util.Collection;
import java.util.List;

import org.joda.time.DateTime;
import org.junit.After;
import org.junit.Before;
import org.junit.Test;

import com.wakaleo.jpt.taxcalculator.InvalidYearException;

public class TaxCalculatorImplTest {

    TaxCalculatorImpl taxCalculator = null;
    
    @Before
    public void prepareTaxCalculator() {
        taxCalculator = new TaxCalculatorImpl();
    }

    @After
    public void cleanupTaxCalculator() {
        taxCalculator = null;
    }

    @Test
    public void shouldUseLowestTaxRateForIncomeBelow38000() throws InvalidYearException {
        double income = 30000;
        double expectedTax = income * 0.195;
        double calculatedTax = taxCalculator.calculateIncomeTax(30000, 2008);
        assertThat(calculatedTax, is(expectedTax));
    }
        
    @Test
    public void lossesShouldNotBeTaxed()  throws InvalidYearException {
        double calculatedTax = taxCalculator.calculateIncomeTax(-10000, 2008);
        assertEquals("Straty nie podlegaj opodatkowaniu", 0, calculatedTax, 0);
    }

    @Test(expected=InvalidYearException.class)
    public void futureYearsShouldBeInvalid() throws InvalidYearException {
        DateTime today = new DateTime();
        int nextYear = today.getYear() + 1;
        double income = 30000;
        taxCalculator.calculateIncomeTax(income, nextYear);
    }
    
    @Test
    public void exceptionShouldIncludeAClearMessage() throws InvalidYearException {
        double income = 30000;
        try {
            taxCalculator.calculateIncomeTax(income, 2100);
            fail("Metoda calculateIncomeTax() powinna wygenerowa wyjtek.");
        } catch (InvalidYearException expected) {
            assertEquals(expected.getMessage(),
                         "Dla roku 2100 nie wyliczono jeszcze adnego podatku.");
        }
    }
    
    @Test
    public void hamcrestAssertEqualToRed(){
        String color = "czerwony";
        assertThat(color, equalTo("czerwony"));
    }
    
    @Test
    public void hamcrestAssertIsRed(){
        String color = "czerwony";
        assertThat(color, is("czerwony"));
    }
    
    @Test
    public void hamcrestAssertIsRedOrBLue(){
        String color = "czerwony";
        assertThat(color, isOneOf("czerwony","niebieski"));
    }

    @Test
    public void hamcrestAssertNotBlue(){
        String color = "czerwony";
        assertThat(color, not("niebieski"));
    }

    @Test
    public void hamcrestAssertNotNull(){
        String color = "czerwony";
        assertThat(color, is(notNullValue()));
    }
    

    @Test
    public void hamcrestAssertNull(){
        String color = null;
        assertThat(color, is(nullValue()));
    }

    @Test
    public void hamcrestAssertHasItem(){
        List<String> colors = new ArrayList<String>();
        colors.add("czerwony");
        colors.add("zielony");
        colors.add("niebieski");
        assertThat(colors, hasItem("niebieski"));
    }

    @Test
    public void hamcrestAssertHasItems(){
        List<String> colors = new ArrayList<String>();
        colors.add("czerwony");
        colors.add("zielony");
        colors.add("niebieski");
        assertThat(colors, hasItems("czerwony","niebieski"));
        assertThat("czerwony", isIn(colors));
    }
    
    @Test
    public void hamcrestAssertHasItemInArray(){
        String[] colors = new String[] {"czerwony","zielony","niebieski"};
        assertThat(colors, hasItemInArray("niebieski"));
    }   
   
    @Test
    public void hamcrestAssertIsIn(){
        String[] colors = new String[] {"czerwony","zielony","niebieski"};
        assertThat("czerwony", isIn(colors));
    }

    
    @Test
    public void hamcrestAssertIsRedOrBlue(){
        String color = "czerwony";
        assertThat(color, anyOf(is("czerwony"),(is("niebieski"))));
        assertThat(color,is(String.class));
    }

    @Test
    public void hamcrestIsClass(){
        List myList = new ArrayList();     
        assertThat(myList,is(Collection.class));
    }
    
    @Test
    public void hamcrestAssertIsBetween10and20(){
        int value = 15;       
        assertThat(value, allOf(greaterThanOrEqualTo(10),
                                lessThanOrEqualTo(20)));
        
        assertTrue("Warto powinna nalee do przedziau od 10 do 20",
                   value >= 10 && value <= 20);
    }
    
    
    @Test
    public void hamcrestAssertNotIsIn(){
        String[] colors = new String[] {"czerwony","zielony","niebieski"};
        String color = "ty";
        assertThat(color, not(isIn(colors)));        
    }       

    public void hamcrestHasItemLessThan(){
        List<Integer> ages = new ArrayList<Integer>();
        ages.add(20);
        ages.add(30);
        ages.add(40);
        assertThat(ages, not(hasItem(lessThan(18))));
    }
}