package view;

import java.io.*;
import java.nio.file.*;
import java.sql.*;
import java.util.*;

import javax.sql.rowset.*;

import util.*;

/**
 * Program wykorzystujący metadane do prezentacji dowolnych tabel bazy danych
 * @version 1.0 2023-08-16
 * @author Cay Horstmann
 */
public class ViewDB2
{
   private CachedRowSet crs;
   private Scanner console;

   record Column(String label, int width, String columnClass) {}
   private List<Column> columns = new ArrayList<>();
   
   public static void main(String[] args)
   {
      new ViewDB2().run();
   }
   
   public void run()
   {
      try (Connection conn = getConnection(readDatabaseProperties()))
      {
         console = new Scanner(System.in);
         DatabaseMetaData meta = conn.getMetaData();
         var tableNames = new ArrayList<String>();

         try (ResultSet mrs = meta.getTables(null, null, null, new String[] { "TABLE" }))
         {
            while (mrs.next())
               tableNames.add(mrs.getString(3));
         }
         System.out.println("Wybierz tabelę");
         String selected = Choices.choose(console, tableNames);
         showTable(selected, conn);
         executeCommands();         
         crs.acceptChanges(conn);
      }
      catch (SQLException e)
      {
         for (Throwable t : e)
            t.printStackTrace();
      }      
      catch (IOException e)
      {
         e.printStackTrace();
      }
   }      
      
   /**
    * Przygotowuje pola tekstowe do prezentacji nowej tabeli i wyświetla 
    * zawartość jej pierwszego rekordu.
    * @param tableName nazwa prezentowanej tabeli
    * @param conn połączenie z bazą danych
    */
   public void showTable(String tableName, Connection conn) throws SQLException
   {
      try (Statement stat = conn.createStatement();
            ResultSet result = stat.executeQuery("SELECT * FROM " + tableName))
      {
         // skopiowanie do buforowanego zbioru wierszy
         RowSetFactory factory = RowSetProvider.newFactory();            
         crs = factory.createCachedRowSet();
         crs.setTableName(tableName);
         crs.populate(result);
         
         ResultSetMetaData rsmd = result.getMetaData();
         for (int i = 1; i <= rsmd.getColumnCount(); i++)
         {
            columns.add(new Column(rsmd.getColumnLabel(i), 
                  rsmd.getColumnDisplaySize(i), 
                  rsmd.getColumnClassName(i)));
         }
         
         showNextRow();
      }
   }

   public void executeCommands() throws SQLException
   {
      while (true)
      {
         String selection 
            = Choices.choose(console, "Następny", "Poprzedni", "Edytuj", "Usuń", "Wyjdź");
         switch (selection)
         {
            case "Następny" -> showNextRow();
            case "Poprzedni" -> showPreviousRow();
            case "Edytuj" -> editRow();
            case "Usuń" -> deleteRow();
            default -> { return; }
         }
      }
   }

   /**
    * Wyświetla wiersz bazy danych poprzez wypełnienie wszystkich pól tekstowych 
    * wartościami z odpowiednich kolumn
    */
   public void showRow(ResultSet rs) throws SQLException
   {
      for (int i = 1; i <= columns.size(); i++)
      {
         String format = "%%s [%%%ds]%%n".formatted(columns.get(i - 1).width());
         System.out.printf(format, columns.get(i - 1).label(), 
            rs == null ? "" : rs.getString(i));
      }
   }
   
   /**
    * Przechodzi do poprzedniego wiersza
    */
   public void showPreviousRow() throws SQLException
   {
      if (crs == null || crs.isFirst()) return;
      crs.previous();
      showRow(crs);
   }

   /**
    * Przechodzi do następnego wiersza.
    */
   public void showNextRow() throws SQLException
   {
      if (crs == null || crs.isLast()) return;
      crs.next();
      showRow(crs);
   }

   /**
    * Usuwa bieżący wiersz tabeli.
    */
   public void deleteRow() throws SQLException 
   {
      if (crs == null) return;
      crs.deleteRow();
      if (crs.isAfterLast())
      {         
         if (!crs.last()) crs = null;
      }
      showRow(crs);
   }

   /**
    * Aktualizuje zmodyfikowane dane w bieżącym wierszu zbioru danych
    */
   public void editRow() throws SQLException
   {
      List<String> editableColumns = columns.stream()
            .filter(c -> c.columnClass.equals("java.lang.String"))
            .map(Column::label)
            .toList();
      System.out.println("Wybierz kolumnę");
      String label = Choices.choose(console, editableColumns);
      System.out.print("Nowa wartość:  ");
      String newValue = console.nextLine();
      crs.updateString(label, newValue);
      crs.updateRow();
   }
   
   private Properties readDatabaseProperties() throws IOException
   {
      var props = new Properties();
      try (Reader in = Files.newBufferedReader(Path.of("database.properties")))
      {
         props.load(in);
      }
      String drivers = props.getProperty("jdbc.drivers");
      if (drivers != null) System.setProperty("jdbc.drivers", drivers);
      return props;
   }
   
   /**
    * Zwraca połączenie z bazą danych na podstawie właściwości podanych 
    * w pliku database.properties
    * @return połączenie z bazą danych
    */
   private Connection getConnection(Properties props) throws SQLException
   {
      String url = props.getProperty("jdbc.url");
      String username = props.getProperty("jdbc.username");
      String password = props.getProperty("jdbc.password");

      return DriverManager.getConnection(url, username, password);
   }
}