package preferences;

import java.awt.EventQueue;
import java.awt.event.*;
import java.io.*;
import java.util.prefs.*;
import javax.swing.*;

/**
 * Program testujący ustawienia preferencji. Zapamiętuje
 * położenie i rozmiar ramki i ostatnio otwarty plik.
 * @version 1.10 2018-04-10
 * @author Cay Horstmann
 */
public class ImageViewer
{
   public static void main(String[] args)
   {
      EventQueue.invokeLater(() -> 
         {
         var frame = new ImageViewerFrame();
         frame.setTitle("ImageViewer");
         frame.setDefaultCloseOperation(JFrame.EXIT_ON_CLOSE);
         frame.setVisible(true);
      });
   }
}

/**
 * Przeglądarka plików, która przywraca ustawienia pozycji, rozmiaru i obrazu
 * z preferencji użytkownika oraz aktualizuje preferencje przy zamykaniu programu
 */
class ImageViewerFrame extends JFrame
{
   private static final int DEFAULT_WIDTH = 300;
   private static final int DEFAULT_HEIGHT = 200;
   private String image;
   
   public ImageViewerFrame()
   {
      Preferences root = Preferences.userRoot();
      Preferences node = root.node("/com/horstmann/corejava/ImageViewer");
      // Pobranie pozycji, rozmiaru i tytułu z preferencji
      int left = node.getInt("left", 0);
      int top = node.getInt("top", 0);
      int width = node.getInt("width", DEFAULT_WIDTH);
      int height = node.getInt("height", DEFAULT_HEIGHT);
      setBounds(left, top, width, height);
      image = node.get("image", null);
      var label = new JLabel();
      if (image != null) label.setIcon(new ImageIcon(image));

      addWindowListener(new WindowAdapter()
      {
         public void windowClosing(WindowEvent event)
         {
            node.putInt("left", getX());
            node.putInt("top", getY());
            node.putInt("width", getWidth());
            node.putInt("height", getHeight());
            if (image != null) node.put("image", image);
         }
      });

      // Użycie etykiety do wyświetlania obrazów
      add(label);

      // Element wyboru pliku
      var chooser = new JFileChooser();
      chooser.setCurrentDirectory(new File("."));

      // Pasek menu
      var menuBar = new JMenuBar();
      setJMenuBar(menuBar);

      var menu = new JMenu("Plik");
      menuBar.add(menu);

      var openItem = new JMenuItem("Otwórz");
      menu.add(openItem);
      openItem.addActionListener(event -> {

         // Wyświetla okno dialogowe wyboru pliku
         int result = chooser.showOpenDialog(null);

         // Jeśli plik zostanie wybrany, ustawia go jako ikonę etykiety
         if (result == JFileChooser.APPROVE_OPTION)
         {
            image = chooser.getSelectedFile().getPath();
            label.setIcon(new ImageIcon(image));
         }
      });

      var exitItem = new JMenuItem("Zamknij");
      menu.add(exitItem);
      exitItem.addActionListener(event -> System.exit(0));
   }
}
