package reflection;

import java.lang.annotation.Annotation;
import java.lang.reflect.Method;
import java.util.ArrayList;
import java.util.List;

// tag::findAnnotatedClasses[]
/** Klasa pozwala odnajdywać "wtyczki" lub inne klasy dodatkowe,
 * wykorzystując w tym celu introspekcję oraz adnotacje.
 */
public class PluginsViaAnnotations {

    /**
     * Metoda odnajduje wszystkie klasy podanego pakietu, do których
     * dodano konkretną adnotację (określoną w formie klasy adnotacji).
     */
    public static List<Class<?>> findAnnotatedClasses(String packageName,
        Class<? extends Annotation> annotationClass) throws Exception {

        List<Class<?>> ret = new ArrayList<>();
        String[] clazzNames = ClassesInPackage.getPackageContent(packageName);
        for (String clazzName : clazzNames) {
            if (!clazzName.endsWith(".class")) {
                continue;
            }
            clazzName = clazzName.replace('/', '.').replace(".class", "");
            Class<?> c = null;
            try {
                c = Class.forName(clazzName);
            } catch (ClassNotFoundException ex) {
                System.err.println("Dziwne: klasa " + clazzName + 
                    " ma być dostępna w pakiecie, lecz próba jej pobrania " +
                    " zgłosiła wyjątek ClassNotFoundException: " + ex);
                continue;
            }
            if (c.isAnnotationPresent(annotationClass) &&
                    !ret.contains(c))
                    ret.add(c);
            
        }
        return ret;
    }
    // end::findAnnotatedClasses[]
    
    // tag::findClassesWithAnnotatedMethods[]
    /**
     * Metoda odnajduje wszystkie klasy podanego pakietu zawierające 
     * jakieś metody, do których została dodana określona adnotacja.
     */
    public static List<Class<?>> findClassesWithAnnotatedMethods(String packageName, 
            Class<? extends Annotation> methodAnnotationClass) throws Exception {
        List<Class<?>> ret = new ArrayList<>();
        String[] clazzNames = ClassesInPackage.getPackageContent(packageName);
        for (String clazzName : clazzNames) {
            if (!clazzName.endsWith(".class")) {
                continue;
            }
            clazzName = clazzName.replace('/', '.').replace(".class", "");
            Class<?> c = null;
            try {
                c = Class.forName(clazzName);
                // System.out.println("Wczytano " + c);
            } catch (ClassNotFoundException ex) {
                System.err.println("Dziwne: klasa " + clazzName + 
                    " ma być dostępna w pakiecie, lecz próba jej pobrania " +
                    " zgłosiła wyjątek ClassNotFoundException: " + ex);
                continue;
            }
            for (Method m : c.getDeclaredMethods()) {
                // System.out.printf("Klasa %s metoda: %s\n",
                //     c.getSimpleName(), m.getName());
                if (m.isAnnotationPresent(methodAnnotationClass) &&
                        !ret.contains(c)) {
                    ret.add(c);
                }
            }
        }
        return ret;
    }
    // end::findClassesWithAnnotatedMethods[]
}
