package otherlang;

import java.io.*;
import java.util.logging.Logger;

/**
 * ExecDemo jak wykonać zewnętrzny program (w tym przypadku 
 * systemowy program uniksowy /bin/ls) i odczytać jego wyniki.
 */
// tag::main[]
public class ExecDemoLs {
    
    private static Logger logger =
        Logger.getLogger(ExecDemoLs.class.getSimpleName());

    /** Program, który należy uruchomić. */
    public static final String PROGRAM = "ls"; // "dir" w przypadku Windows.
    /** True, aby zakończyć pętlę. */
    static volatile boolean done = false;

    public static void main(String argv[]) throws IOException {

        final Process p;      // Obiekt process reprezentuje jeden rodzimy proces.
        BufferedReader is;    // Obiekt, w którym będą zapisywane wyniki
                              // wykonywanego procesu.
        String line;
        
        p = Runtime.getRuntime().exec(PROGRAM);

        logger.info("W metodzie main po wywołaniu exec.");

        // Opcjonalne: uruchamiamy wątek oczekujący na zakończenie 
        // procesu. Nie będziemy czekać w metodzie main() - tutaj 
        // ustawiamy jedynie flagę "done" i używamy jej do kontroli
        // działania głównej pętli odczytującej, umieszczonej poniżej.
        Thread waiter = new Thread() {
            public void run() {
                try {
                    p.waitFor();
                } catch (InterruptedException ex) {
                    // OK, po prostu kończymy.
                    return;
                }
                System.out.println("Program został zakończony!");
                done = true;
            }
        };
        waiter.start();

        // Metoda getInputStream zwraca strumień wejściowy (InputStream)
        // skojarzony ze standardowym wyjściem uruchomionego procesu p
        // (i na odwrót). Użyjemy go do utworzenia obiektu
        // BufferedReader, dzięki czemu będziemy mogli odczytywać wiersze
        // tekstu przy użyciu metody readLine().
        is = new BufferedReader(new InputStreamReader(p.getInputStream()));

        while (!done && ((line = is.readLine()) != null))
            System.out.println(line);
        
        logger.info("W metodzie main po zakończeniu odczytu.");

        return;
    }
}
// end::main[]
