// package com.darwinsys.io;
public class FileSaver {
    
    private enum State {
        /** Stan przed użyciem i po nim. */
        AVAILABLE,
        /** Stan w trakcie używania. */
        INUSE
    }

    private State state;
    private final Path inputFile;
    private final Path tmpFile;
    private final Path backupFile;

    private OutputStream mOutputStream;
    private Writer mWriter;

    public FileSaver(Path inputFile) throws IOException {
        // Krok 1. Tworzymy plik tymczasowy w odpowiednim miejscu; musi
        // się on znajdować na tym samym dysku co pierwotny plik, by uniknąć
        // późniejszych problemów z wyczerpaniem miejsca na dysku.
        this.inputFile = inputFile;
        tmpFile = Path.of(inputFile.normalize() + ".tmp");
        Files.createFile(tmpFile);
        tmpFile.toFile().deleteOnExit();
        backupFile = Path.of(inputFile.normalize() + ".bak");
        state = State.AVAILABLE;
    }

    /**
     * Metoda zwraca odwołanie do używanego obiektu File, zachęcając
     * tym samym do jego wielokrotnego użycia (obiekty File 
     * są niezmienne, więc takie rozwiązanie jest umiarkowanie
     * bezpieczne). Tak może wyglądać typowy przykład użycia tej
     * metody:
     * <pre>
     * if (fileSaver == null ||
     * !(fileSaver.getFile().equals(file))) {
     * fileSaver = new FileSaver(file);
     * }
     * </pre>
     * @return obiekt File zapisywanego pliku.
     */
    public Path getFile() {
        return inputFile;
    }

    /** Metoda zwraca plik wyjściowy, w którym klient powinien
     * zapisywać dane użytkownika.
     * @return Obiekt OutputStream, który w celu zapewnienia rozsądnej
     *   wydajności działania powinien zostać użyty do utworzenia 
     *   obiektu OutputStream.
     * @throws IOException, jeśli nie uda się utworzyć pliku tymczasowego.
     */
    public OutputStream getOutputStream() throws IOException {
        
        if (state != State.AVAILABLE) {
            throw new IllegalStateException("Obiekt FileSaver nie został utworzony!");
        }
        mOutputStream = Files.newOutputStream(tmpFile);
        state = State.INUSE;
        return mOutputStream;
    }

    /** Metoda zwraca plik wyjściowy, w którym klient powinien 
     * zapisywać dane użytkownika.
     * @return BufferedWriter do zapisu danych w nowym pliku.
     * @throws IOException, jeśli nie uda się utworzyć pliku tymczasowego.
     */
    public Writer getWriter() throws IOException {
        
        if (state != State.AVAILABLE) {
            throw new IllegalStateException("Obiekt FileSaver nie został utworzony!");
        }
        mWriter = Files.newBufferedWriter(tmpFile);
        state = State.INUSE;
        return mWriter;
    }

    /** Metoda zamyka plik wyjściowy i zmienia jego nazwę na docelową.
     * @throws IOException, jeśli pojawią się jakiekolwiek problemy.
     */
    public void finish() throws IOException {

        if (state != State.INUSE) {
            throw new IllegalStateException("Obiekt FileSaver nie jest używany.");
        }

        // Przed próbą zmiany nazwy upewniamy się, że oba strumienie są zamknięte.
        if (mOutputStream != null) {
            mOutputStream.close();
        }
        if (mWriter != null) {
            mWriter.close();
        }

        // Usuwamy wcześniejszy plik kopii bezpieczeństwa, 
        // jeśli taki istnieje.
        Files.deleteIfExists(backupFile);

        // Zmieniamy nazwę wcześniejszego pliku z danymi użytkownika na 
        // wczesnieszaNazwa.bak.
        // CHYBA ŻE to jest nowy plik.
        if (Files.exists(inputFile) &&
            Files.move(inputFile, backupFile) == null) {
            throw new IOException(
                "Nie udało się zmienić nazwy pliku na nazwę pliku " +
                "kopii bezpieczeństwa: " + backupFile);
        }

        // Zmieniamy nazwę pliku tymczasowego na docelową nazwę pliku 
        // z danymi użytkownika.
        if (Files.move(tmpFile, inputFile) == null) {
            throw new IOException("Nie udało się zmienić nazwy pliku " + 
                                                "tymczasowego na docelową.");
        }
        state = State.AVAILABLE;
    }
}