package dir_file;

import java.io.IOException;
import java.nio.file.Files;
import java.nio.file.Path;
import java.util.regex.Pattern;


// tag::main[]
/** Klasa hermetyzująca operację filtrowania dla klasy Find.
 * Póki co zaimplementowane zostały jedynie metody set*Filter(). 
 * W przypadku złożonych wyszukiwań, takich jak to poniżej, powinna 
 * to być rzeczywista struktura danych (może LinkedList<FilterOp> 
 * lub Tree):
 *	-n "*.html" -a \( -size < 0 -o mtime < 5 \).
 */
public class FindFilter {
    private enum SizeMode {GT, EQ, LT};
    SizeMode sizeMode;
    Find.Conjunction conj;
    long size;
    String name;
    Pattern nameRE;
    boolean debug = false;

    void setSizeFilter(String sizeFilter) {
        System.out.println("FindFilter.setSizeFilter()");
        sizeMode = SizeMode.EQ;
        char c = sizeFilter.charAt(0);
        if (c == '+') {
            sizeMode = SizeMode.GT;
            sizeFilter = sizeFilter.substring(1);
        } else {
            if (c == '-') {
                sizeMode = SizeMode.LT;
                sizeFilter = sizeFilter.substring(1);
            }
        }
        size = Long.parseLong(sizeFilter);
    }
    
    /** Połączenie z użyciem operatora. */
    public void addConjunction(Find.Conjunction conj) {
        System.out.println("FindFilter.addConjunction()");
        if (this.conj != null) {
            throw new IllegalArgumentException(
                "Ta wersja obsługuje tylko jedno połączenie");
        }
        this.conj = conj;
    }

    /** Konwertuje podany wzorzec znaków wieloznacznych na postać
     * używaną wewnętrznie - czyli wyrażenie regularne.
     */
    void setNameFilter(String nameToFilter) {
        nameRE = makeNameFilter(nameToFilter);
    }
    
    Pattern makeNameFilter(String name) {
        StringBuilder sb = new StringBuilder('^');
        for (char c : name.toCharArray()) {
            switch(c) {
                case '.':	sb.append("\\."); break;
                case '*':	sb.append(".*"); break;
                case '?':	sb.append('.'); break;
                // W wyrażeniach regularnych niektóre znaki mają specjalne
                // znaczenie i trzeba je odpowiednio zabezpieczyć.
                case '[':	sb.append("\\["); break;
                case ']':	sb.append("\\]"); break;
                case '(':	sb.append("\\("); break;
                case ')':	sb.append("\\)"); break;
                default:	sb.append(c); break;
            }
        }
        sb.append('$');
        if (debug) {
            System.out.println("RE=\"" + sb + "\".");
        }
        // Należałoby przechwytywać wyjątek PatternException i ponownie 
        // go zgłaszać, by zapewnić lepszą diagnostykę.
        return Pattern.compile(sb.toString());
    }

    /** Realizacja filtrowania. W tym przypadku wyłącznie na podstawie 
     * nazwy, wielkości lub nazwy i wielkości łącznie.
     */
    public boolean accept(Path p) throws IOException {
        if (debug) {
            System.out.println("FindFilter.accept(" + p + ")");
        }
        
        if (nameRE != null) {
            return nameRE.matcher(p.getFileName().toString()).matches();
        }

        // Obsługa wielkości.
        if (sizeMode != null) {
            long sz = Files.size(p);
            switch (sizeMode) {
            case EQ:
                return (sz == size);
            case GT:
                return (sz > size);
            case LT:
                return (sz < size);
            }
        }

        // Cała reszta.
        return false;
    }
    
    public String getName() {
        return name;
    }
}
// end::main[]
