package chat;

import java.awt.BorderLayout;
import java.awt.Font;
import java.io.BufferedReader;
import java.io.IOException;
import java.io.InputStreamReader;
import java.io.PrintWriter;
import java.net.Socket;
import java.util.concurrent.*;

import javax.swing.JButton;
import javax.swing.JFrame;
import javax.swing.JLabel;
import javax.swing.JOptionPane;
import javax.swing.JPanel;
import javax.swing.JTextArea;
import javax.swing.JTextField;

/** 
 * <p>
 * Simple Chat Room GUI.
 * Writing a Chat Room seemed to be one of many obligatory rites (or wrongs)
 * of passage for Java experts in the early days.</p>
 * <p>
 * This one is a toy because it doesn't have much of a protocol, which
 * means we can't query the server as to who's logged in,
 * or anything fancy like that. However, it works OK for small groups.</p>
 * <p>
 * Uses client socket w/ two Threads (main and one constructed),
 * one for reading and one for writing.</p>
 * <p>
 * Server multiplexes messages back to all clients.</p>
 * @author Ian Darwin
 */
// tag::main[]
public class ChatClient extends JFrame {

    private static final long serialVersionUID = -3686334002367908392L;
    private static final String userName = 
        System.getProperty("user.name", "Bezimienny użytkownik");
    /** Czy użytkownik jest zalogowany? */
    protected boolean loggedIn;
    /** Główna ramka aplikacji. */
    protected JFrame cp;
    /** Domyślny numer portu. */
    protected static final int PORTNUM = ChatProtocol.PORTNUM;
    /** Używany numer portu. */
    protected int port;
    /** Gniazdo. */
    protected Socket sock;
    /** Obiekt PrintWriter służący do wysyłania wierszy tekstu. */
    protected PrintWriter pw;
    /** TextField - pole wejściowe. */
    protected JTextField tf;
    /** TextArea - pole do wyświetlania pogawędki. */
    protected JTextArea ta;
    /** Przycisk Login. */
    protected JButton loginButton;
    /** Przycisk Logout. */
    protected JButton logoutButton;
    /** Tytuł wyświetlany na pasku nagłówka. */
    final static String TITLE = "ChatClient: Prosty klient pogawędek Iana Darwina";

    final Executor threadPool = Executors.newSingleThreadExecutor();

    /** Przygotowanie graficznego interfejsu użytkownika. */
    public ChatClient() {
        cp = this;
        cp.setTitle(TITLE);
        cp.setLayout(new BorderLayout());
        port = PORTNUM;
        
        // Interfejs użytkownika
        ta = new JTextArea(14, 80);
        ta.setEditable(false);		// Tylko do odczytu.
        ta.setFont(new Font("Monospaced", Font.PLAIN, 11));
        cp.add(BorderLayout.NORTH, ta);

        JPanel p = new JPanel();

        // Przycisk do logowania.
        p.add(loginButton = new JButton("Logowanie"));
        loginButton.setEnabled(true);
        loginButton.requestFocus();
        loginButton.addActionListener(e -> {
                login();
                loginButton.setEnabled(false);
                logoutButton.setEnabled(true);
                tf.requestFocus();	// Określenie miejsca wprowadzania.
        });

        // Przycisk do wylogowania.
        p.add(logoutButton = new JButton("Wyloguj się"));
        logoutButton.setEnabled(false);
        logoutButton.addActionListener(e -> {
                logout();
                loginButton.setEnabled(true);
                logoutButton.setEnabled(false);
                loginButton.requestFocus();
        });

        p.add(new JLabel("Treść wiadomości:"));
        tf = new JTextField(40);
        tf.addActionListener(e -> {
                if (loggedIn) {
                    pw.println(ChatProtocol.CMD_BCAST+tf.getText());
                    tf.setText(""); 
                }
        });
        p.add(tf);

        cp.add(BorderLayout.SOUTH, p);

        cp.setDefaultCloseOperation(JFrame.EXIT_ON_CLOSE);
        cp.pack();
    }

    protected String serverHost = "localhost";

    /** Podłączamy się do pogawędki. */
    public void login() {
        /** Obiekt BufferedReader do odczytu z gniazda. */
        BufferedReader is;
        
        showStatus("Logujemy się!");
        if (loggedIn)
            return;
        try {
            sock = new Socket(serverHost, port);
            is = new BufferedReader(new InputStreamReader(sock.getInputStream()));
            pw = new PrintWriter(sock.getOutputStream(), true);
            showStatus("Mamy połączenie");

            // Udajemy logowanie - na razie nie potrzeba hasła.
            pw.println(ChatProtocol.CMD_LOGIN + userName);

            loggedIn = true;

        } catch(IOException e) {
            showStatus("Nie można nawiązać połączenia z " + 
                serverHost + "/" + port + ": " + e);
            cp.add(new Label("Nie można uzyskać gniazda: " + e));
            return;
        }

        // Tworzymy i uruchamiamy czytelnika: z serwera do wielowierszowego
        // pola tekstowego. Aby uniknąć blokowania, używamy wątku.
        Runnable readerThread = new Runnable() {
            public void run() {
                String line;
                try {
                    while (loggedIn && ((line = is.readLine()) != null))
                        ta.append(line + "\n");
                } catch(IOException e) {
                    showStatus("Połączenie zostało utracone!\n" + e);
                    return;
                }
            }
        };
        threadPool.execute(readerThread);
    }

    /** Wychodzimy stąd, Scotty, nie ma tu żadnych inteligentnych form 
     * życia! */
    public void logout() {
        if (!loggedIn)
            return;
        loggedIn = false;
        try {
            if (sock != null)
                sock.close();
        } catch (IOException ign) {
            // No i co z tego?
        }
    }

    public void showStatus(String message) {
        System.out.println(message);
    }

    private void warn(String message) {
        JOptionPane.showMessageDialog(this, message);
    }

    /** Metoda główna umożliwiająca uruchomienie programu jako 
     * normalnej aplikacji. 
     */
    public static void main(String[] args) {
        ChatClient room101 = new ChatClient();
        room101.pack();
        room101.setVisible(true);
    }
}
// end::main[]
