package com.darwinsys.tools;

import java.io.FileNotFoundException;
import java.io.IOException;
import java.net.ConnectException;
import java.net.MalformedURLException;
import java.net.SocketException;
import java.net.URI;
import java.net.UnknownHostException;
import java.net.http.HttpClient;
import java.net.http.HttpRequest;
import java.net.http.HttpResponse;
import java.net.http.HttpClient.Redirect;
import java.net.http.HttpClient.Version;
import java.net.http.HttpResponse.BodyHandlers;

/**
 * Minimalistyczny program do sprawdzania odnośników; sprawdza jeden lub kilka odnośników podanych w wierszu
 * poleceń, robiąc to uważnie, lecz nie rekurencyjnie.
 * Typowy sposób użucia: java -cp darwinsys-api.jar com.darwinsys.net.KwikLinkChecker [-v] url [...]
 * @author Ian Darwin
 */
public class KwikLinkChecker {
	
	static boolean verbose;
	HttpClient client;

	public static void main(String[] args) {
		KwikLinkChecker checker = new KwikLinkChecker();
		for (String arg : args) {
			if (arg.equals("-v")) {
				verbose = true;
				continue;
			}
			if (arg.startsWith("-")) {
				System.out.println("Nieprawidłowy argument: " + arg);
				continue;
			}
			LinkStatus stat = checker.check(arg);
			if (verbose || !stat.ok)
				System.out.println(stat.message);
		}
	}

	KwikLinkChecker() {
		client = HttpClient.newBuilder()
			.followRedirects(Redirect.NORMAL)
			.version(Version.HTTP_1_1)
			.build();
	}
	
	// tag::main[]
	/**
	 * Metoda sprawdza jeden odnośnik HTTP. Nie działa rekurencyjnie.
	 * Zwraca obiekt LinkStatus z logicznym polem określającym status
	 * wykonanego sprawdzenia i nazwą pliku lub komentarzem o błędzie 
	 * w polu message. Końcówka tej metody jest jednym z niewielu 
	 * miejsc, w których konieczna jest cała sekwencja różnych 
	 * klauzul catch, niezbędnych do prawidłowego działania programu.
	 * @param urlString Odnośnik do sprawdzenia.
	 * @return Status odnośnika.
	 */
	@SuppressWarnings("exports")
	public LinkStatus check(String urlString) {
		try {
			HttpResponse<String> resp = client.send(
				HttpRequest.newBuilder(URI.create(urlString))
				.header("User-Agent", getClass().getName())
				.GET()
				.build(), 
				BodyHandlers.ofString());

			// Pobieramy odpowiedź.
			if (resp.statusCode() == 200) {
				System.out.println(resp.body());
			} else {
				System.out.printf("BŁĄD: kod statusu %d, żądanie %s\n",
					resp.statusCode(), urlString);
			}

			switch (resp.statusCode()) {
			case 200:
				return new LinkStatus(true, urlString);
			case 403:
				return new LinkStatus(false,"403: " + urlString );
			case 404:
				return new LinkStatus(false,"404: " + urlString );
			}
			return new LinkStatus(true, urlString);
		} catch (IllegalArgumentException | MalformedURLException e) {
			// JDK zgłasza wyjątek IllegalArgumentException,
			// jeśli na podstawie adresu URL nie uda się określić 
			// komputera docelowego. 
			return new LinkStatus(false, "Nieprawidłowy adres URL: " + urlString);
		} catch (UnknownHostException e) {
			return new LinkStatus(false, "Nieprawidłowy lub nieaktywny komputer: " + urlString);
		} catch (FileNotFoundException e) {
			return new LinkStatus(false,"NIE ODNALEZIONO (404) " + urlString);
		} catch (ConnectException e) {
			return new LinkStatus(false, "Serwer nie działa: " + urlString);
		} catch (SocketException e) {
			return new LinkStatus(false, e + ": " + urlString);
		} catch (IOException e) {
			return new LinkStatus(false, e.toString()); // Dołączamy URL.
		} catch (Exception e) {
			return new LinkStatus(false, urlString + ": " + e);
		}
	}
	// end::main[]
}