package dir_file;

import java.io.File;
import java.nio.file.*;
import java.nio.file.attribute.FileTime;
import java.io.IOException;
import java.util.Date;

/**
 * Wyświetla informacje o statusie pliku
 * @author Ian F. Darwin, http://www.darwinsys.com/
 */
// tag::main[]
public class FileStatus {
    public static void main(String[] argv) throws IOException {

        // Upewniamy się, czy w argv[0] podano nazwę pliku (lub coś innego).
        if (argv.length == 0) {
            System.err.println("Sposób użycia: FileStatus nazwaPliku");
            System.exit(1);
        }
        for (String a : argv) {
            status(a);
        }
    }

    public static void status(String fileName) throws IOException {
        System.out.println("---" + fileName + "---");

        // Tworzymy obiekt Path dla danego pliku.
        Path p = Path.of(fileName);

        // Sprawdzamy, czy plik istnieje.
        if (!Files.exists(p)) {
            System.out.println("Nie znaleziono pliku.");
            System.out.println();	// Pusty wiersz.
            return;
        }
        // Wyświetlamy pełną nazwę.
        System.out.println("Pełna nazwa " + p.normalize());
        // Jeśli to możliwe, to wyświetlamy katalog nadrzędny. 
        Path parent = p.getParent();
        if (parent != null) {
            System.out.println("Katalog nadrzędny: " + parent);
        }
        // Sprawdzamy, czy odczyt pliku jest możliwy.
        if (Files.isReadable(p)) {
            System.out.println(fileName + " - odczyt możliwy.");
        }
        // Sprawdzamy, czy zapis pliku jest możliwy.
        if (Files.isWritable(p)) {
            System.out.println(fileName + " - zapis możliwy.");
        }

        // Sprawdzamy, czy mamy do czynienia z plikiem, katalogiem, czy czymś innym.
        // Jeśli to plik, to wyświetlamy jego wielkość.
        if (Files.isRegularFile(p)) {
            // Wyświetlamy informację o wielkości i ewentualne o typie pliku.
            System.out.printf("Wielkość pliku to %d bajtów, typ zawartości pliku %s\n",
                    Files.size(p),
                    Files.probeContentType(p));
        } else if (Files.isDirectory(p)) {
            System.out.println("To jest katalog.");
        } else {
            System.out.println("Nie wiem - to nie jest ani plik, ani katalog.");
        }

        // Informacje o dacie modyfikacji.
        final FileTime d = Files.getLastModifiedTime(p);
        System.out.println("Czas ostatniej modyfikacji: " + d);

        System.out.println(); // Pusty wiersz pomiędzy wyświetlanymi informacjami.
    }
    // end::main[]

    public static void statusLegacy(String fileName) throws IOException {
        System.out.println("---" + fileName + "---");

        // Tworzymy obiekt File dla danego pliku.
        File f = new File(fileName);

        // Sprawdzamy, czy plik istnieje.
        if (!f.exists()) {
            System.out.println("Nie znaleziono pliku.");
            System.out.println();	// Pusty wiersz.
            return;
        }
        // Wyświetlamy pełną nazwę.
        System.out.println("Pełna nazwa " + f.getCanonicalPath());
        // Jeśli to możliwe, to wyświetlamy katalog nadrzędny. 
        String p = f.getParent();
        if (p != null) {
            System.out.println("Katalog nadrzędny: " + p);
        }
        // Sprawdzamy, czy odczyt pliku jest możliwy.
        if (f.canRead()) {
            System.out.println("Odczyt pliku jest możliwy.");
        }
        // Sprawdzamy, czy zapis pliku jest możliwy.
        if (f.canWrite()) {
            System.out.println("Zapis pliku jest możliwy.");
        }
        // Wyświetlamy informacje o czasie modyfikacji.
        Date d = new Date(f.lastModified());
        System.out.println("Czas ostatniej modyfikacji: " + d);

        // Sprawdzamy, czy mamy do czynienia z plikiem, katalogiem, czy czymś innym.
        // Jeśli to plik, to wyświetlamy jego wielkość.
        if (f.isFile()) {
            // Wyświetlamy informację o wielkości pliku.
            System.out.println("Wielkość pliku to " + f.length() + " bajtów.");
        } else if (f.isDirectory()) {
            System.out.println("To jest katalog.");
        } else {
            System.out.println("Nie wiem - to nie jest ani plik, ani katalog.");
        }

        System.out.println();	// Pusty wiersz pomiędzy wyświetlanymi informacjami.
    }
}
