/**
 * @brief Wypisz liczbę operacji resetowania systemu do urządzenia szeregowego.
 */
#include <stdbool.h>
#include "stm32f0xx_nucleo.h"
#include "stm32f0xx.h"

const char message1[] = "This system had been rebooted ";   // Część 1. komunikatu.
const char message2[] = " times\r\n";                       // Część 2. komunikatu.
const char many[] = "many";                                 // Słowo "many" (wiele).
uint32_t resetCount __attribute__((section(".config.keep"))) = 0;    // # Liczba wykonanych operacji resetowania.
int current; // Znak w wysyłanym komunikacie.

UART_HandleTypeDef uartHandle;      // Inicjacja interfejsu UART.

/**
  * @brief Funkcja ta jest wykonywana w razie wystąpienia błędu.
  *
  * Jedyne, co robi, to miganie diodą LED.
  */
void Error_Handler(void)
{
    /* Włącz diodę LED2. */
    HAL_GPIO_WritePin(LED2_GPIO_PORT, LED2_PIN, GPIO_PIN_SET);

    while (true)
    {
    // Przełącz stan diody LED2.
        HAL_GPIO_TogglePin(LED2_GPIO_PORT, LED2_PIN);
        HAL_Delay(1000);        // Poczekaj 1 sekundę.
    }
}
/**
 * Wyślij znak do UART.
 *
 * @param ch Wysyłany znak.
 */
void myPutchar(const char ch)
{
    // W wierszu tym pobiera się i zapisuje wartość bitu UART_FLAG_TXE w czasie
    // wywołania. Wartość ta zmienia się, więc jeśli zatrzymasz program na znajdującym się
    // poniżej wierszu z instrukcją "while", wartość będzie równa 0, bo znika
    //szybciej, niż można na nią spojrzeć.
    int result __attribute__((unused)) =
        (uartHandle.Instance->ISR & UART_FLAG_TXE);

    // Blokuj wykonanie do chwili, gdy ustawiona zostanie flaga pustego rejestru nadawczego (TXE).
    while ((uartHandle.Instance->ISR & UART_FLAG_TXE) == 0)
        continue;

    uartHandle.Instance->TDR = ch;     // Wyślij znak do interfejsu UART.
}

/**
 * Wyślij łańcuch znaków do interfejsu UART.
 *
 * @param msg Wysyłany komunikat.
 */
static void myPuts(const char* const msg)
{
    for (unsigned int i = 0; msg[i] != '\0'; ++i) {
        myPutchar(msg[i]);
    }
}

/**
 * Zainicjuj LED2 (po to, by można było zasygnalizować błąd miganiem na czerwono).
 */
void led2_Init(void)
{
    // Inicjacja zegara LED.
    LED2_GPIO_CLK_ENABLE();

    GPIO_InitTypeDef GPIO_LedInit;      // Inicjacja LED.
    // Zainicjuj LED.
    GPIO_LedInit.Pin = LED2_PIN;
    GPIO_LedInit.Mode = GPIO_MODE_OUTPUT_PP;
    GPIO_LedInit.Pull = GPIO_PULLUP;
    GPIO_LedInit.Speed = GPIO_SPEED_FREQ_HIGH;
    HAL_GPIO_Init(LED2_GPIO_PORT, &GPIO_LedInit);
}

/**
 * Zainicjuj interfejs UART2 w celu wyprowadzania danych.
 */
void uart2_Init(void)
{
    // Inicjacja interfejsu UART.
    //  UART2 -- ten połączony z interfejsem USB  programatora ST-LINK.
    uartHandle.Instance = USART2;
    uartHandle.Init.BaudRate = 9600;                    // Prędkość 9600 bodów.
    uartHandle.Init.WordLength = UART_WORDLENGTH_8B;    // 8 bitów na znak.
    uartHandle.Init.StopBits = UART_STOPBITS_1;         // 1 bit stopu.
    uartHandle.Init.Parity = UART_PARITY_NONE;          // Brak kontroli parzystości.
    uartHandle.Init.Mode = UART_MODE_TX_RX;             // Wysyłanie i odbiór.
    uartHandle.Init.HwFlowCtl = UART_HWCONTROL_NONE;    // Brak sprzętowego sterowania przepływem.

    // Poddaj strumień przychodzący nadpróbkowaniu.
    uartHandle.Init.OverSampling = UART_OVERSAMPLING_16;

    // Nie stosuj jednej próbki na bit.
    uartHandle.Init.OneBitSampling = UART_ONE_BIT_SAMPLE_DISABLE;

    // Żadnych zaawansowanych funkcji.
    uartHandle.AdvancedInit.AdvFeatureInit = UART_ADVFEATURE_NO_INIT;
    /*
     * Dla tych z Was, którzy łączą się przez emulator terminala: powyższe parametry
     * przekładają się na format transmisji 9600,8,N,1.
     */

    if (HAL_UART_Init(&uartHandle) != HAL_OK)
    {
        Error_Handler();
    }
}
/**
 * Aktualizacja licznika resetCount.
 *
 * W języku C byłaby to instrukcja ++resetCount. Ponieważ mamy do czynienia
 * z pamięcią flash, operacja ta jest dużo trudniejsza.
 */
static HAL_StatusTypeDef updateCounter(void) {
    HAL_FLASH_Unlock(); // Pozwól na modyfikację pamięci flash.
    uint32_t newResetCount = resetCount + 1;    // Następna wartość licznika operacji resetowania.

    uint32_t pageError = 0;     // Wskazanie błędu w operacji wymazywania.
    // Poleć systemowi pamięci flash wymazać zmienną resetCount (i resztę strony).
    FLASH_EraseInitTypeDef eraseInfo = { 
        .TypeErase = FLASH_TYPEERASE_PAGES,     // Zamierzamy wymazać jedną stronę.
        .PageAddress = (uint32_t)&resetCount,   // Początek strony.
        .NbPages = 1                            // Jedna strona do wymazania.
    };
    
    // Wymaż stronę i pobierz wynik operacji.
    HAL_StatusTypeDef result = HAL_FLASHEx_Erase(&eraseInfo, &pageError);
    if (result != HAL_OK) {
        HAL_FLASH_Lock();
        return (result);
    }

    // Zaprogramuj w pamięci nową wartość licznika operacji resetowania.
    result = HAL_FLASH_Program(FLASH_TYPEPROGRAM_WORD, 
            (uint32_t)&resetCount, newResetCount);
    if (result != HAL_OK) {
        HAL_FLASH_Lock();
        return (result);
    }
    return (result);
}

int main(void)
{
    HAL_Init(); // Zainicjuj sprzęt.
    led2_Init();
    uart2_Init();

    myPuts(message1);

    HAL_StatusTypeDef status = updateCounter();

    switch (status) {
        case HAL_FLASH_ERROR_NONE:
            // Nic nie rób -- to poprawna wartość.
            break;
        case HAL_FLASH_ERROR_PROG:
            myPuts("HAL_FLASH_ERROR_PROG");
            break;
        case HAL_FLASH_ERROR_WRP:
            myPuts("HAL_FLASH_ERROR_WRP");
            break;
        default:
            myPuts("**unknown error code**");
            break;
    }
    // Sztuczka zastosowana po to, byśmy nie musieli pisać funkcji konwertującej liczbę całkowitą na  łańcuch ASCII. 
    if (resetCount < 10)
        myPutchar('0'+ resetCount);
    else 
        myPuts("many");

    myPuts(message2);

    for (;;) {
        continue;       // Nic nie rób.
    }
}

/**
 * Tajemna funkcja wywoływana przez warstwę HAL w celu faktycznego
 * zainicjowania interfejsu UART. W tym przypadku musimy 
 * przestawić piny UART w tryb alternatywny, by działały jako
 * linie UART, a nie GPIO.
 *
 * @note: Działa tylko z interfejsem UART2, tym połączonym z konwerterem
 * USB/port szeregowy.
 *
 * @param uart Dane UART.
 */
void HAL_UART_MspInit(UART_HandleTypeDef* uart)
{
    GPIO_InitTypeDef GPIO_InitStruct;
    if(uart->Instance == USART2)
    {
        /* Włączenie zegara urządzenia peryferyjnego. */
        __HAL_RCC_USART2_CLK_ENABLE();

        /*
         * Konfigurowanie bitów GPIO interfejsu USART2.
         * PA2     ------> USART2_TX
         * PA3     ------> USART2_RX
         */
        GPIO_InitStruct.Pin = GPIO_PIN_2|GPIO_PIN_3;
        GPIO_InitStruct.Mode = GPIO_MODE_AF_PP;
        GPIO_InitStruct.Pull = GPIO_NOPULL;
        GPIO_InitStruct.Speed = GPIO_SPEED_FREQ_LOW;
        // Funkcja alternatywna -- związana z UART.
        GPIO_InitStruct.Alternate = GPIO_AF1_USART2;
        HAL_GPIO_Init(GPIOA, &GPIO_InitStruct);
    }

}

/**
 * Tajemna funkcja wywoływana przez warstwę HAL w celu przywrócenia
 * urządzenia UART do stanu niezainicjowanego. Nigdy tego nie robimy, ale umieściliśmy tu tę
 * funkcję gwoli kompletności.
 *
 * @note: Działa tylko z interfejsem UART2, tym połączonym z konwerterem
 * USB/port szeregowy.
 *
 * @param uart Dane UART.
 */
void HAL_UART_MspDeInit(UART_HandleTypeDef* uart)
{
    if(uart->Instance == USART2)
    {
        /* Wyłączenie  zegara urządzenia peryferyjnego. */
        __HAL_RCC_USART2_CLK_DISABLE();

        /*
         * Konfigurowanie bitów GPIO interfejsu USART2.
         * PA2     ------> USART2_TX
         * PA3     ------> USART2_RX
         */
        HAL_GPIO_DeInit(GPIOA, GPIO_PIN_2|GPIO_PIN_3);
    }
}
