/**
 * @brief Wypisz "hello world" do portu szeregowego.
 */
#include <stdbool.h>
#include "stm32f0xx_nucleo.h"
#include "stm32f0xx.h"

const char hello[] = "Hello World!\r\n";   // Wysyłany komunikat.
int current; // Znak w wysyłanym komunikacie.

UART_HandleTypeDef uartHandle;      // Inicjacja interfejsu UART.

/**
  * @brief Funkcja ta jest wykonywana w razie wystąpienia błędu.
  *
  * Jedyne, co robi, to miganie diodą LED.
  */
void Error_Handler(void)
{
    /* Włącz diodę LED2. */
    HAL_GPIO_WritePin(LED2_GPIO_PORT, LED2_PIN, GPIO_PIN_SET);

    while (true)
    {
    // Przełącz stan diody LED2.
        HAL_GPIO_TogglePin(LED2_GPIO_PORT, LED2_PIN);
        HAL_Delay(1000);        // Poczekaj 1 sekundę.
    }
}
/**
 * Wyślij znak do interfejsu UART.
 *
 * @param ch Wysyłany znak.
 */
void myPutchar(const char ch)
{
    // W wierszu tym pobiera się i zapisuje wartość bitu UART_FLAG_TXE w czasie
    // wywołania. Wartość ta zmienia się, więc jeśli zatrzymasz program na znajdującym się
    // lponiżej wierszu z instrukcją "while", wartość będzie równa 0, bo znika
    // szybciej, niż można na nią spojrzeć.
    int result __attribute__((unused)) =
        (uartHandle.Instance->ISR & UART_FLAG_TXE);

    // Blokuj wykonanie do chwili, gdy ustawiona zostanie flaga pustego rejestru nadawczego (TXE).
    while ((uartHandle.Instance->ISR & UART_FLAG_TXE) == 0)
        continue;

    uartHandle.Instance->TDR = ch;     // Wyślij znak do interfejsu UART.
}

/**
 * Zainicjuj LED2 (po to, by można było zasygnalizować błąd miganiem na czerwono).
 */
void led2_Init(void)
{
    // Inicjacja zegara LED.
    LED2_GPIO_CLK_ENABLE();

    GPIO_InitTypeDef GPIO_LedInit;      // Inicjacja LED.
    // Zainicjuj LED.
    GPIO_LedInit.Pin = LED2_PIN;
    GPIO_LedInit.Mode = GPIO_MODE_OUTPUT_PP;
    GPIO_LedInit.Pull = GPIO_PULLUP;
    GPIO_LedInit.Speed = GPIO_SPEED_FREQ_HIGH;
    HAL_GPIO_Init(LED2_GPIO_PORT, &GPIO_LedInit);
}

/**
 * Zainicjuj interfejs UART2 w celu wyprowadzania danych.
 */
void uart2_Init(void)
{
    // Inicjacja interfejsu UART.
    // UART2 -- ten połączony z interfejsem USB  programatora ST-LINK.
    uartHandle.Instance = USART2;
    uartHandle.Init.BaudRate = 9600;                    // Prędkość 9600 bodów.
    uartHandle.Init.WordLength = UART_WORDLENGTH_8B;    // 8 bitów na znak.
    uartHandle.Init.StopBits = UART_STOPBITS_1;         // 1 bit stopu.
    uartHandle.Init.Parity = UART_PARITY_NONE;          // Brak kontroli parzystości.
    uartHandle.Init.Mode = UART_MODE_TX_RX;             // Wysyłanie i odbiór.
    uartHandle.Init.HwFlowCtl = UART_HWCONTROL_NONE;    // Brak sprzętowego sterowania przepływem.

    // Poddaj strumień przychodzący nadpróbkowaniu.
    uartHandle.Init.OverSampling = UART_OVERSAMPLING_16;

    // Nie stosuj jednej próbki na bit.
    uartHandle.Init.OneBitSampling = UART_ONE_BIT_SAMPLE_DISABLE;

    // Żadnych zaawansowanych funkcji.
    uartHandle.AdvancedInit.AdvFeatureInit = UART_ADVFEATURE_NO_INIT;
    /*
     * Dla tych z Was, którzy łączą się przez emulator terminala: powyższe parametry
     * przekładają się na format transmisji 9600,8,N,1.
     */

    if (HAL_UART_Init(&uartHandle) != HAL_OK)
    {
        Error_Handler();
    }
}

int main(void)
{
    HAL_Init(); //Zainicjuj sprzęt.
    led2_Init();
    uart2_Init();


    // Kontynuuj wysyłanie komunikatu przez długi czas.
    for (;;) {
        // Wysyłaj znak po znaku.
        for(current = 0; hello[current] != '\0'; ++current) {
            myPutchar(hello[current]);
        }
        HAL_Delay(500);
    }
}

/**
 * Tajemna funkcja wywoływana przez warstwę HAL w celu faktycznego
 * zainicjowania interfejsu UART. W tym przypadku musimy
 * przestawić piny UART na tryb alternatywny, by działały jako
 * linie UART, a nie GPIO.
 *
 * @note: Działa tylko z interfejsem UART2, tym połączonym z konwerterem
 * USB/port szeregowy.
 *
 * @param uart Dane UART.
 */
void HAL_UART_MspInit(UART_HandleTypeDef* uart)
{
    GPIO_InitTypeDef GPIO_InitStruct;
    if(uart->Instance == USART2)
    {
        /* Włączenie zegara urządzenia peryferyjnego.  */
        __HAL_RCC_USART2_CLK_ENABLE();

        /*
         * Konfigurowanie bitów GPIO interfejsu USART2.
         * PA2     ------> USART2_TX
         * PA3     ------> USART2_RX
         */
        GPIO_InitStruct.Pin = GPIO_PIN_2|GPIO_PIN_3;
        GPIO_InitStruct.Mode = GPIO_MODE_AF_PP;
        GPIO_InitStruct.Pull = GPIO_NOPULL;
        GPIO_InitStruct.Speed = GPIO_SPEED_FREQ_LOW;
        // Funkcja alternatywna -- związana z UART.
        GPIO_InitStruct.Alternate = GPIO_AF1_USART2;
        HAL_GPIO_Init(GPIOA, &GPIO_InitStruct);
    }

}

/**
 * Tajemna funkcja wywoływana przez warstwę HAL w celu przywrócenia
 * urządzenia UART do stanu niezainicjowanego. Nigdy tego nie robimy, ale umieściliśmy tu tę
 * funkcję gwoli kompletności.
 *
 * @note: Działa tylko z interfejsem UART2, tym połączonym z konwerterem
 * USB/port szeregowy.
 *
 * @param uart Dane UART.
 */
void HAL_UART_MspDeInit(UART_HandleTypeDef* uart)
{
    if(uart->Instance == USART2)
    {
        /* Wyłączenie  zegara urządzenia peryferyjnego */
        __HAL_RCC_USART2_CLK_DISABLE();

        /*
         * Konfigurowanie bitów GPIO interfejsu USART2
         * PA2     ------> USART2_TX
         * PA3     ------> USART2_RX
         */
        HAL_GPIO_DeInit(GPIOA, GPIO_PIN_2|GPIO_PIN_3);
    }
}

