package rozdzial09;

import java.awt.*;
import java.util.Arrays;

public interface IColor {
    /**
     * Zamienia liczbę HEX na RGB
     *
     * @param hexa String - liczba typu 0xFFFFFF podawana jest jako
     *             "FFFFFF" bez 0x
     * @return String - wartość RGB
     */
    static String HexToRGB(String hexa) {
        Integer R = Integer.valueOf(hexa.substring(0, 2), 16);
        Integer G = Integer.valueOf(hexa.substring(2, 4), 16);
        Integer B = Integer.valueOf(hexa.substring(4, 6), 16);
        return ("RGB:[" + R + "," + G + "," + B + "]");
    }

    static int[] HexToRGB2(String hexa) {
        Integer R = Integer.valueOf(hexa.substring(0, 2), 16);
        Integer G = Integer.valueOf(hexa.substring(2, 4), 16);
        Integer B = Integer.valueOf(hexa.substring(4, 6), 16);
        return new int[]{R, G, B};
    }

    /**
     * Zamienia liczbę HEX na RGB
     *
     * @param hexa int - podajemy liczbę typu 0xFFFFFF
     * @return String - wartość RGB
     */
    static String HexToRGB(int hexa) {
        int R = (hexa >> 16) & 0xff;
        int G = (hexa >> 8) & 0xff;
        int B = hexa & 0xff;
        return ("RGB:[" + R + "," + G + "," + B + "]");
    }

    /**
     * Zamienia wartosci R, G, B na liczbę HEX
     *
     * @param r int - składnik czerwony
     * @param g int - składnik zielony
     * @param b int - składnik niebieski
     * @return String - waartość HEX
     */
    static String RGBToHex(int r, int g, int b) {
        String r1 = Integer.toString(r, 16);
        String g1 = Integer.toString(g, 16);
        String b1 = Integer.toString(b, 16);
        if (r1.length() == 1) {
            r1 = "0" + r1;
        }
        if (g1.length() == 1) {
            g1 = "0" + g1;
        }
        if (b1.length() == 1) {
            b1 = "0" + b1;
        }
        return "0x" + r1 + g1 + b1;
    }

    /**
     * Zamienia wartosci R, G, B na liczbę int stosowaną w Javie
     *
     * @param r int - składnik czerwony;
     * @param g int - składnik zielony
     * @param b int - składnik niebieski
     * @return int - wartość RGB przeliczona na int;
     */
    static int RGBToInt(int r, int g, int b) {
        String r1 = Integer.toString(r, 16);
        String g1 = Integer.toString(g, 16);
        String b1 = Integer.toString(b, 16);
        if (r1.length() == 1) {
            r1 = "0" + r1;
        }
        if (g1.length() == 1) {
            g1 = "0" + g1;
        }
        if (b1.length() == 1) {
            b1 = "0" + b1;
        }
        String cont = "" + r1 + g1 + b1;
        int i = Integer.valueOf(cont, 16);
        return (i | 0x00 << 16 | 0x00 << 8 | 0x00);
    }

    /**
     * Przelicza liczbę int na R, G, B
     *
     * @param rgb int - liczba int oznaczająca kolor;
     * @return String - wartość RGB
     */
    static String IntToRGB(int rgb) {
        int R = (rgb >> 16) & 0xff;
        int G = (rgb >> 8) & 0xff;
        int B = (rgb >> 0) & 0xff;
        return ("RGB:[" + R + "," + G + "," + B + "]");
    }

    /**
     * Zamienia wartosci R, G, B, A na liczbę int stosowaną w Javie
     *
     * @param r int - składnik czerwony;
     * @param g int - składnik zielony
     * @param b int - składnik niebieski
     * @param a int - składnik alfa
     * @return int - wartość RGB przeliczona na int;
     */
    static int ARGBToInt(int a, int r, int g, int b) {
        String a1 = Integer.toString(a, 16);
        String r1 = Integer.toString(r, 16);
        String g1 = Integer.toString(g, 16);
        String b1 = Integer.toString(b, 16);
        if (a1.length() == 1) {
            a1 = "0" + a1;
        }
        if (r1.length() == 1) {
            r1 = "0" + r1;
        }
        if (g1.length() == 1) {
            g1 = "0" + g1;
        }
        if (b1.length() == 1) {
            b1 = "0" + b1;
        }
        String cont = "" + a1 + r1 + g1 + b1;
        return Integer.valueOf(cont, 16);
    }

    /**
     * Przelicza liczbę int na R, G, B, A
     *
     * @param rgba int - liczba int oznaczająca kolor;
     * @return String - wartość RGBA
     */
    static String IntToARGB(int rgba) {
        int A = (rgba >> 24) & 0xff;
        int R = (rgba >> 16) & 0xff;
        int G = (rgba >> 8) & 0xff;
        int B = (rgba >> 0) & 0xff;
        return ("ARGB:[" + A + "," + R + "," + G + "," + B + "]");
    }

    //

    /**
     * funkcja zaokragla podaną liczbę do określonej liczby miejsc po przecinku
     *
     * @param num double - liczba do zaokrąglenia
     * @param dec int - liczba miejsc po przecinku;
     * @return double - zwraca zaokrągloną liczbę
     */
    static double roundToDecimal(double num, int dec) {
        int multi = (int) Math.pow(10, dec);
        int temp = (int) Math.round(num * multi);
        return (double) temp / multi;
    }

    static double max(double... values) {
        int len = values.length;
        Arrays.sort(values);
        return values[len - 1];
    }

    static double min(double... values) {
        Arrays.sort(values);
        return values[0];
    }

    /**
     * Przekształca HSL na RGB	 *
     *
     * @param h     Hue - wstopniach 0 - 360.
     * @param s     Saturation - %.
     * @param l     Lumanance - %.
     * @param alpha - 0.0 - 1.0	 *
     * @returns Color
     */
    static Color HSLToRGB(float h, float s, float l, float alpha) {
        if (s < 0.0f || s > 100.0f) {
            String message = "Saturation poza zakresem";
            throw new IllegalArgumentException(message);
        }
        if (l < 0.0f || l > 100.0f) {
            String message = "Luminance poza zakresem";
            throw new IllegalArgumentException(message);
        }
        if (alpha < 0.0f || alpha > 1.0f) {
            String message = "Alpha poza zakresem";
            throw new IllegalArgumentException(message);
        }
        h = h % 360.0f;
        h /= 360f;
        s /= 100f;
        l /= 100f;
        float q;
        if (l < 0.5)
            q = l * (1 + s);
        else
            q = (l + s) - (s * l);
        float p = 2 * l - q;
        float r = Math.max(0, HueToRGB(p, q, h + (1.0f / 3.0f)));
        float g = Math.max(0, HueToRGB(p, q, h));
        float b = Math.max(0, HueToRGB(p, q, h - (1.0f / 3.0f)));
        r = Math.min(r, 1.0f);
        g = Math.min(g, 1.0f);
        b = Math.min(b, 1.0f);
        return new Color(r, g, b, alpha);
    }

    /**
     * Przekształca RGB na HSL	 *
     *
     * @return tablica [H,L,S].
     */
    static float[] RGBToHSL(Color color) {
        float[] rgb = color.getRGBColorComponents(null);
        float r = rgb[0];
        float g = rgb[1];
        float b = rgb[2];
        //	Minimum and Maximum RGB values are used in the HSL calculations
        float min = Math.min(r, Math.min(g, b));
        float max = Math.max(r, Math.max(g, b));
        //  Calculate the Hue
        float h = 0;
        if (max == min)
            h = 0;
        else if (max == r)
            h = ((60 * (g - b) / (max - min)) + 360) % 360;
        else if (max == g)
            h = (60 * (b - r) / (max - min)) + 120;
        else if (max == b)
            h = (60 * (r - g) / (max - min)) + 240;
        //  Calculate the Luminance
        float l = (max + min) / 2;
        //  Calculate the Saturation
        float s;
        if (max == min)
            s = 0;
        else if (l <= .5f)
            s = (max - min) / (max + min);
        else
            s = (max - min) / (2 - max - min);
        return new float[]{h, s * 100, l * 100};
    }

    static float HueToRGB(float p, float q, float h) {
        if (h < 0)
            h += 1;
        if (h > 1)
            h -= 1;
        if (6 * h < 1) {
            return p + ((q - p) * 6 * h);
        }
        if (2 * h < 1) {
            return q;
        }
        if (3 * h < 2) {
            return p + ((q - p) * 6 * ((2.0f / 3.0f) - h));
        }
        return p;
    }

    static int HSBToRGB(float hue, float saturation, float brightness) {
        return Color.HSBtoRGB(hue, saturation, brightness);
    }

    static float[] RGBToHSB(int r, int g, int b) {
        return Color.RGBtoHSB(r, g, b, null);
    }
}
