/* $Id$ */
/* Copyright R 2002 George Reese, Imaginet */
package org.dasein.persist;

// Opracowa  George Reese dla potrzeb ksiki:
// Java. Aplikacje bazodanowe. Najlepsze rozwizania: J2EE
// Przenis na bibliotek kodu digital@jwt George Reese


import java.sql.Connection;
import java.sql.PreparedStatement;
import java.sql.ResultSet;
import java.sql.SQLException;
import java.util.HashMap;

import javax.naming.InitialContext;
import javax.naming.NamingException;
import javax.sql.DataSource;

/**
 * Narzdzie suce do automatycznego generowania niepowtarzalnych liczb. Ta klasa
 * komunikuje sie z baz danych co <code>MAX_KEYS</code> da
 * nowego zarodka dla generowanych liczb. Ta klasa charakteryzuje si bezpieczestwem
 * wtkw, co oznacza, e wiele wtkw moe bezpiecznie da od niej niepowtarzalnych liczb.
 * Klasa zapewnia take bezpieczestwo wielu procesw. Inaczej mwic wiele komputerw
 * moe jednoczenie generowa niepowtarzalne wartoci i maj one gwarancj,
 * e wartoci te bd niepowtarzalne na przestrzeni wszystkich aplikacji. Jedynym wymaganiem
 * jest uycie we wszystkich aplikacjach tego samego algorytmu generowania liczb
 * i pobierania zarodkw z bazy danych. W celu uzyskania dostpu do bazy danych,
 * klasa wymaga wartoci pola systemowego
 * <code>org.dasein.persist.SequencerDSN</code>. Pole to naley ustawi na
 * warto nazwy rda danych zapewniajcego poczenia z baz danych
 * za pomoc tabeli <code>Sequencer</code>. Tabel t mona utworzy
 * za pomoc nastpujcej instrukcji <code>CREATE</code>:
 * <span class="code">
 * CREATE TABLE Sequencer (
 *     name        VARCHAR(20)     NOT NULL,
 *     seed        BIGINT UNSIGNED NOT NULL,
 *     lastUpdate  BIGINT UNSIGNED NOT NULL,
 *     PRIMARY KEY ( name, lastUpdate )
 * );
 * </span>
 * <br/>
 * Data ostatniej modyfikacji $Date$
 * @wersja $Revision$
 * @autor George Reese
 */
public class Sequencer {
    /**
     * Maksymalna liczba kluczy, ktre mona bezpieczniw wygenerowa bez koniecznoci
     * komunikacji z baz danych. Liczba ta powinna mie mniejsz warto dla      * aplikacji klienckich i innych programw o krtkotrwaym dziaaniu. Natomiast
     * wiksz warto naley ustawi dla aplikacji dziaajcych przez duszy czas. 
     * Trzeba pamita, e dla wszystkich aplikacji wykorzystujcych ten sam sekwencer      * naley ustawi t sam warto <code>MAX_KEYS</code>.
     */
    static private final long    MAX_KEYS   = 1000000L;
    /**
     * Wszystkie sekwencery zaadowane do pamici.
     */
    static private final HashMap sequencers = new HashMap();

    /**
     * Metoda sprawdza, czy dla sekwencji o okrelonej nazwie      * wygenerowano sekwencer. Jeeli tak si nie stao, metoda wytwarza egzemparz sekwencera.
     * Wielokrotne wywoanie tej metody z t sam nazw zawsze powoduje
     * zwrcenie tego samego obiektu sekwencera. Dla uzyskania najlepszej wydajnoci
     * klasy powinny zapisywa odwoania do sekwencera zaraz po jego uzyskaniu.
     * Czynno ta pozwala unikn niepotrzebnych obcie zwizanych z przeszukiwaniem      * konstrukcji <code>HashMap</code>.
     * @parametr name nazwa danego sekwencera
     * @zwraca sekwencer o okreslonej nazwie
     */
    static public final Sequencer getInstance(String name) {
        synchronized( sequencers ) {
            if( !sequencers.containsKey(name) ) {
                Sequencer seq = new Sequencer(name);

                sequencers.put(name, seq);
                return seq;
            }
            else {
                return (Sequencer)sequencers.get(name);
            }
        }
    }

    /**
     * Nazwa sekwencera.
     */
    private String name     = null;
    /**
     * Zarodek wykorzystywany przez sekwencer do generowania identyfikatorw.
     */
    private long   seed     = -1L;
    /**
     * Bieca sekwencja w ramach tego zarodka sekwencera.
     */
    private long   sequence = 0L;

    /**
     * Tworzy nowy sekwencer o okrelonej nazwie.
     * @parametr nom nazwa sekwencera
     */
    private Sequencer(String nom) {
        super();
        name = nom;
    }

   /**
     * Kod SQL sucy do utworzenia nowej sekwencji w bazie danych.
     */
    static private final String CREATE_SEQ =
        "INSERT INTO Sequencer ( name, seed, lastUpdate ) " +
        "VALUES ( ?, ?, ? )";
    /**
     * Staa dla paramatru name.
     */
    static private final int INS_NAME   = 1;
    /**
     * Staa dla parametru zarodek.
     */
    static private final int INS_SEED   = 2;
    /**
     * Staa dla parametru lastUpdate
     */
    static private final int INS_UPDATE = 3;

    /**
     * Tworzy now pozycj w bazie danych dla tej sekwencji. Metoda zgasza
     * bd, jeeli dwa wtki jednoczenie prbuj utworzy
     * sekwencj. Taki stan nie powinien si zdarzy, jeeli sekwencja w bazie danych
     * zostanie utworzona przed zainstalowaniem aplikacji. Problemw mona
     * unikn przez sprawdzenie, czy zgoszono waciwy wyjtek SQL XOPEN SQLState      * dla duplikatw kluczy. Niestety przy tym podejciu wystpuje due      * prawdopodobiestwo bdw, ze wzgldu na brak spjnoci przy      * zgaszaniu waciwych wartoci XOPEN SQLState dla sterownikw JDBC.
     * @parametr conn Wykorzystywane poczenie JDBC
     * @zglasza java.sql.SQLException wystpi bd bazy danych
     */
    private void create(Connection conn) throws SQLException {
        PreparedStatement stmt = null;
        ResultSet rs = null;

        try {
            stmt = conn.prepareStatement(CREATE_SEQ);
            stmt.setString(INS_NAME, name);
            stmt.setLong(INS_SEED, 0L);
            stmt.setLong(INS_UPDATE, System.currentTimeMillis());
            if( stmt.executeUpdate() != 1 ) {
                throw new SQLException("Nie wprowadzono wiersza.");
            }
            seed = 0L;
        }
        finally {
            if( rs != null ) {
                try { rs.close(); }
                catch( SQLException e ) { }
            }
            if( stmt != null ) {
                try { stmt.close(); }
                catch( SQLException e ) { }
            }
        }
    }

    /**
     * Nazwa rda danych do wykorzystania, jeeli nie skonfigurowano jej
     * we waciwociach systemu.
     */
    static private final String DEFAULT_DSN = "jdbc/kyra";
    /**
     * Nazwa waciwoci systemowej zawierajcej nazw rda danych.
     */
    static private final String DSN_PROP    = "org.dasein.persist.DSN";

    /**
     * Generuje nowy niepowtarzalny identyfikator. Algorytm generowania      * jest nastpujcy:<br/>
     * <i>niepowtarzalny identyfikator</i> = <i>zarodek</i> pomnoony przez
     * <i>maksymaln liczb kluczy na zarodek</i> plus to <i>sekwencja zarodka</i>
     * <br/>
     * Metoda inkrementuje nastpnie sekwencj zarodka w celu umoliwienia
     * wygenerowania nastpneho identyfikatora. Jeeli wygenerowane ID
     * spowoduje wyczerpanie si zarodka, nastpi pobranie nowego zarodka z
     * bazy danych.
     * @zwraca niepowtarzalny identyfikator
     * @zglasza org.dasein.persist.PersistenceException wystpi bd magazynu danych
     * podczas generowania liczby
     */
    public synchronized long next() throws PersistenceException {
        Connection conn = null;

        // Jeeli zarodek wynosi -1 lub zarodek nie wystarcza do generowania kluczy,
        // pobranie nowego zarodka z bazy danych
        if( (seed == -1L) || ((sequence + 1) >= MAX_KEYS) ) {
            try {
                String dsn = System.getProperty(DSN_PROP, DEFAULT_DSN);
                InitialContext ctx = new InitialContext();
                DataSource ds = (DataSource)ctx.lookup(dsn);

                conn = ds.getConnection();
                reseed(conn);
            }
            catch( SQLException e ) {
                throw new PersistenceException(e);
            }
            catch( NamingException e ) {
                throw new PersistenceException(e);
            }
            finally {
                if( conn != null ) {
                    try { conn.close(); }
                    catch( SQLException e ) { }
                }
            }
        }
        // zwikszenie wartoci sekwencji dla nastpnego klucza
        sequence++;
        // nastpny klucz dla tego sekwencera
        return ((seed * MAX_KEYS) + sequence);
    }

    /**
     * Instrukcja SQL pobierajca z bazy danych zarodek sekwencji.
     */
    static private final String FIND_SEQ =
        "SELECT seed, lastUpdate " +
        "FROM Sequencer " +
        "WHERE name = ?";
    /**
     * Staa dla parametru name.
     */
    static private final int SEL_NAME   = 1;
    /**
     * Staa dla kolumny seed.
     */
    static private final int SEL_SEED   = 1;
    /**
     * Staa dla kolumny lastUpdate.
     */
    static private final int SEL_UPDATE = 2;
    /**
     * Instrukcja SQL zwikszajca warto zarodka w bazie danych.
     */
    static private String UPDATE_SEQ =
        "UPDATE Sequencer " +
        "SET seed = ?, " +
        "lastUpdate = ? " +
        "WHERE name = ? AND lastUpdate = ?";
    /**
     * Staa dla paramatru seed.
     */
    static private final int UPD_SEED         = 1;
    /**
     * Staa dla parametru lastUpdate
     */
    static private final int UPD_SET_UPDATE   = 2;
    /**
     * Staa dla parametru name.
     */
    static private final int UPD_NAME         = 3;
    /**
     * Staa dla parametru lastUpdate.
     */
    static private final int UPD_WHERE_UPDATE = 4;

    /**
     * Pobiera z bazy danych nastpny zarodek dla tej sekwencji.
     * @parametr conn poczenie z baz danych
     * @zglasza java.sql.SQLException wystpi bd bazy danych
     */
    private void reseed(Connection conn) throws SQLException {
        PreparedStatement stmt = null;
        ResultSet rs = null;

        try {
            // Podtrzymuj t ptl tak dugo, jak wystpuj bdy wspbienoci
            do {
                stmt = conn.prepareStatement(FIND_SEQ);
                stmt.setString(SEL_NAME, name);
                rs = stmt.executeQuery();
                if( !rs.next() ) {
                    // nie ma taiej sekwencji, utwrz j
                    {
                        // zamknicie zasobw
                        try { rs.close(); }
                        catch( SQLException e ) { }
                        rs = null;
                        try { stmt.close(); }
                        catch( SQLException e ) { }
                        stmt = null;
                    }
                    create(conn);
                }
                else {
                    long ts;

                    seed = rs.getLong(SEL_SEED) + 1L;
                    ts = rs.getLong(SEL_UPDATE);
                    {
                        // zamknicie zasobw
                        try { rs.close(); }
                        catch( SQLException e ) { }
                        rs = null;
                        try { stmt.close(); }
                        catch( SQLException e ) { }
                        stmt = null;
                    }
                    // zwikszenie zarodka w bazie danych
                    stmt = conn.prepareStatement(UPDATE_SEQ);
                    stmt.setLong(UPD_SEED, seed);
                    stmt.setLong(UPD_SET_UPDATE, System.currentTimeMillis());
                    stmt.setString(UPD_NAME, name);
                    stmt.setLong(UPD_WHERE_UPDATE, ts);
                    if( stmt.executeUpdate() != 1 ) {
                        // kto zaktualizowa baz danych! sprbuj ponownie!
                        seed = -1L;
                    }
                }
            } while( seed == -1L );
            sequence = -1L;
        }
        finally {
            if( rs != null ) {
                try { rs.close(); }
                catch( SQLException e ) { }
            }
            if( stmt != null ) {
                try { stmt.close(); }
                catch( SQLException e ) { }
            }
        }   
    }
}
