/* $Id$ */
/* Copyright R 2002 George Reese, Imaginet */
package org.dasein.persist;

// Opracowa  George Reese dla potrzeb ksiki:
// Java. Aplikacje bazodanowe. Najlepsze rozwizania: J2EE
// Przenis na bibliotek kodu digital@jwt George Reese


import java.io.InputStream;
import java.io.Reader;
import java.math.BigDecimal;
import java.sql.Array;
import java.sql.Blob;
import java.sql.Clob;
import java.sql.Connection;
import java.sql.Date;
import java.sql.PreparedStatement;
import java.sql.Ref;
import java.sql.ResultSet;
import java.sql.ResultSetMetaData;
import java.sql.SQLException;
import java.sql.SQLWarning;
import java.sql.Time;
import java.sql.Timestamp;
import java.util.ArrayList;
import java.util.Calendar;
import java.util.Iterator;

/**
 * Instrukcja logiczna przekazywana do aplikacji jako abstrakcja
 * instrukcji preparowanej z puli.
 * <br/>
 * Data ostatniej modyfikacji $Date$
 * @wersja $Revision$
 * @autor George Reese
 */
public class PooledStatement implements PreparedStatement {
    /**
     * Poczenie logiczne odpowiadajce tej instrukcji.
     */
    private Connection        connection = null;
    /**
     * Lista procesw nasuchujcych zainteresowanych zamkniciem tej instrukcji.
     */
    private ArrayList         listeners  = new ArrayList();
    /**
     * Instrukcja SQL odpowiadajca tej instrukcji preparowanej.
     */
    private String            sql        = null;
    /**
     * Poczenie fizyczne.
     */
    private PreparedStatement statement  = null;

    /**
     * Tworzy now instrukcj puli powizan z okrelonym poczeniem
     * logicznym i instrukcj preparowan.
     * @parametr conn poczenie logiczne z baz danych
     * @parametr sql Instrukcja SQL, ktr reprezentuje instrukcja preparowana
     * @parametr stmt fizyczna instrukcja preparowana
     */
    public PooledStatement(Connection conn, String sql,
                           PreparedStatement stmt) {
        super();
        this.sql = sql;
        statement = stmt;
    }

    /**
     * Metoda delegowana do okrelonej instrukcji preparowanej.
     * @zglasza java.sql.SQLException wystpi bd bazy danych
     */
    public void addBatch() throws SQLException {
        validate();
        statement.addBatch();
    }

    /**
     * Metoda delegowana do okrelonej instrukcji preparowanej.
     * @parametr sql nieuywany
     * @zglasza java.sql.SQLException wystpi bd bazy danych
     */
    public void addBatch(String sql) throws SQLException {
        validate();
        statement.addBatch(sql);
    }

    /**
     * Dodaje proces nasuchujcy do listy procesw odpowiadajcych tej instrukcji.
     * @parametr lstnr proces nasuchujcy do dodania
     */
    public void addStatementEventListener(StatementEventListener lstnr) {
        synchronized( listeners ) {
            listeners.add(lstnr);
        }
    }

    /**
     * Metoda delegowana do okreslonej instrukcji preparowanej.
     * @zglasza java.sql.SQLException wystpi bd bazy danych
     */
    public void cancel() throws SQLException {
        validate();
        statement.cancel();
    }

    /**
     * Metoda delegowana do okrelonej instrukcji preparowanej.
     * @zglasza java.sql.SQLException wystpi bd bazy danych
     */
    public void clearBatch() throws SQLException {
        validate();
        statement.clearBatch();
    }

    /**
     * Metoda delegowana do okrelonej instrukcji preparowanej.
     * @zglasza java.sql.SQLException wystpi bd bazy danych
     */
    public void clearParameters() throws SQLException {
        validate();
        statement.clearParameters();
    }

    /**
     * Metoda delegowana do okrelonej instrukcji preparowanej.
     * @zglasza java.sql.SQLException wystpi bd bazy danych
     */
    public void clearWarnings() throws SQLException {
        validate();
        statement.clearWarnings();
    }

    /**
     * Zamyka poczenie logiczne i informuje wszystkie procesy nasuchujce
     * o tym fakcie, dziki czemu instrukcj fizyczn mona zwrci do puli.
     * Wszystkie parametry i ostrzeenia powizane z instrukcj      * s zerowane.
     * @zglasza java.sql.SQLException wystpi bd bazy danych
     */
    public void close() throws SQLException {
        validate();
        statement.clearParameters();
        statement.clearWarnings();
        statement = null;
        synchronized( listeners ) {
            Iterator it = listeners.iterator();

            while( it.hasNext() ) {
                StatementEventListener lstnr;

                lstnr = (StatementEventListener)it.next();
                lstnr.statementClosed(new StatementEvent(sql, statement));
            }
        }
    }

    /**
     * Metoda delegowana do okrelonej instrukcji preparowanej.
     * @zwraca true jeeli wykonanie wygenerowao wyniki
     * @zglasza java.sql.SQLException wystpi bd bazy danych
     */
    public boolean execute() throws SQLException {
        validate();
        return statement.execute();
    }

    /**
     * Metoda delegowana do okrelonej instrukcji preparowanej.
     * @parametr sql niezbdny dla interfejsu <code>Statement</code>
     * @zwraca true jeeli wykonanie wygenerowao wyniki
     * @zglasza java.sql.SQLException wystpi bd bazy danych
     */
    public boolean execute(String sql) throws SQLException {
        validate();
        return statement.execute(sql);
    }

    /**
     * Metoda delegowana do okrelonej instrukcji preparowanej.
     * @zwraca tablic licznikw aktualizacji
     * @zglasza java.sql.SQLException wystpi bd bazy danych
     */
    public int[] executeBatch() throws SQLException {
        validate();
        return statement.executeBatch();
    }

    /**
     * Metoda delegowana do okrelonej instrukcji preparowanej
     * @zwraca the query results
     * @zglasza java.sql.SQLException wystpi bd bazy danych
     */
    public ResultSet executeQuery() throws SQLException {
        validate();
        return statement.executeQuery();
    }

    /**
     * Metoda delegowana do okrelonej instrukcji preparowanej.
     * @parametr sql nonsense
     * @zwraca the query results
     * @zglasza java.sql.SQLException wystpi bd bazy danych
     */
    public ResultSet executeQuery(String sql) throws SQLException {
        validate();
        return statement.executeQuery(sql);
    }
    
    /**
     * Metoda delegowana do okrelonej instrukcji preparowanej
     * @zwraca liczb zaktualizowanych wierszy
     * @zglasza java.sql.SQLException wystpi bd bazy danych
     */
    public int executeUpdate() throws SQLException {
        validate();
        return statement.executeUpdate();
    }

    /**
     * Metoda delegowana do okrelonej instrukcji preparowanej.
     * @parametr sql bez znaczenia
     * @zwraca liczb zaktualizowanych wierszy
     * @zglasza java.sql.SQLException wystpi bd bazy danych
     */
    public int executeUpdate(String sql) throws SQLException {
        validate();
        return statement.executeUpdate(sql);
    }

    /**
     * Udostpnie poczenie logiczne zwizane z t instrukcj      * @zwraca poczenie dla instrukcji
     * @zglasza java.sql.SQLException wystpi bd bazy danych
     */
    public Connection getConnection() throws SQLException {
        validate();
        return connection;
    }

    /**
     * Metoda delegowana do okrelonej instrukcji preparowanej
     * @zwraca  kierunek pobierania
     * @zglasza java.sql.SQLException wystpi bd bazy danych
     */
    public int getFetchDirection() throws SQLException {
        validate();
        return statement.getFetchDirection();
    }

    /**
     * Metoda delegowana do okrelonej instrukcji preparowanej.
     * @zwraca rozmiar pobierania
     * @zglasza java.sql.SQLException wystpi bd bazy danych
     */
    public int getFetchSize() throws SQLException {
        validate();
        return statement.getFetchSize();
    }

    /**
     * Metoda delegowana do okrelonej instrukcji preparowanej
     * @zwraca maksymalny rozmiar pola
     * @zglasza java.sql.SQLException wystpi bd bazy danych
     */
    public int getMaxFieldSize() throws SQLException {
        validate();
        return statement.getMaxFieldSize();
    }

    /**
     * Metoda delegowana do okrelonej instrukcji preparowanej.
     * @zwraca maksymaln liczb wierszy
     * @zglasza java.sql.SQLException wystpi bd bazy danych
     */
    public int getMaxRows() throws SQLException {
        validate();
        return statement.getMaxRows();
    }

    /**
     * Metoda delegowana do okrelonej instrukcji preparowanej
     * @zwraca metadane zestawu wynikw
     * @zglasza java.sql.SQLException wystpi bd bazy danych
     */
    public ResultSetMetaData getMetaData() throws SQLException {
        validate();
        return statement.getMetaData();
    }

    /**
     * Metoda delegowana do okrelonej instrukcji preparowanej.
     * @zwraca wicej wynikw!
     * @zglasza java.sql.SQLException wystpi bd bazy danych
     */
    public boolean getMoreResults() throws SQLException {
        validate();
        return statement.getMoreResults();
    }
    
    /**
     * Metoda delegowana do okrelonej instrukcji preparowanej
     * @zwraca limit czasu dla zapytania
     * @zglasza java.sql.SQLException wystpi bd bazy danych
     */
    public int getQueryTimeout() throws SQLException {
        validate();
        return statement.getQueryTimeout();
    }

    /**
     * Metoda delegowana do okrelonej instrukcji preparowanej.
     * @zwraca ostatni zestaw wynikw
     * @zglasza java.sql.SQLException wystpi bd bazy danych
     */
    public ResultSet getResultSet() throws SQLException {
        validate();
        return statement.getResultSet();
    }

    /**
     * Metoda delegowana do okrelonej instrukcji preparowanej.
     * @zwraca wspbieno dla zestaww wynikw
     * @zglasza java.sql.SQLException wystpi bd bazy danych
     */
    public int getResultSetConcurrency() throws SQLException {
        validate();
        return statement.getResultSetConcurrency();
    }

    /**
     * Metoda delegowana do okrelonej instrukcji preparowanej.
     * @zwraca typ zestaww wynikw
     * @zglasza java.sql.SQLException wystpi bd bazy danych
     */
    public int getResultSetType() throws SQLException {
        validate();
        return statement.getResultSetType();
    }

    /**
     * Metoda delegowana do okrelonej instrukcji preparowanej
     * @zwraca liczb zaktualizowanych wierszy
     * @zglasza java.sql.SQLException wystpi bd bazy danych
     */
    public int getUpdateCount() throws SQLException {
        validate();
        return statement.getUpdateCount();
    }

    /**
     * Metoda delegowana do okrelonej instrukcji preparowanej
     * @zwraca ostrzeenia
     * @zglasza java.sql.SQLException wystpi bd bazy danych
     */
    public SQLWarning getWarnings() throws SQLException {
        validate();
        return statement.getWarnings();
    }

    /**
     * @zwraca true jeeli nie ma adnych instrukcji
     */
    public boolean isClosed() throws SQLException {
        return (statement == null);
    }

    /**
     * Usuwa proces nasuchujcy z listy.
     * @parametr lstnr proces nasuchujcy do usunicia
     */
    public void removeStatementEventListener(StatementEventListener lstnr) {
        synchronized( listeners ) {
            listeners.remove(lstnr);
        }
    }

    /**
     * Metoda delegowana do okrelonej instrukcji preparowanej
     * @parametr idx indeks do przypisania wartoci
     * @parametr arr wartoci tablicy
     * @zglasza java.sql.SQLException wystpi bd bazy danych
     */
    public void setArray(int idx, Array arr) throws SQLException {
        validate();
        statement.setArray(idx, arr);
    }

    /**
     * Metoda delegowana do okrelonej instrukcji preparowanej
     * @parametr idx indeks do przypisania wartoci
     * @parametr is strumie wejciowy
     * @parametr len dugo strumienia
     * @zglasza java.sql.SQLException wystpi bd bazy danych
     */
    public void setAsciiStream(int idx, InputStream is, int len)
        throws SQLException {
        validate();
        statement.setAsciiStream(idx, is, len);
    }

    /**
     * Metoda delegowana do okrelonej instrukcji preparowanej
     * @parametr idx indeks do przypisania wartoci
     * @parametr d dua warto dziesitna
     * @zglasza java.sql.SQLException wystpi bd bazy danych
     */
    public void setBigDecimal(int idx, BigDecimal d) throws SQLException {
        validate();
        statement.setBigDecimal(idx, d);
    }

    /**
     * Metoda delegowana do okrelonej instrukcji preparowanej
     * @parametr idx indeks do przypisania wartoci
     * @parametr is strumie wejciowy
     * @parametr len dugo strumienia wejciowego
     * @zglasza java.sql.SQLException wystpi bd bazy danych
     */
    public void setBinaryStream(int idx, InputStream is, int len)
        throws SQLException {
        validate();
        statement.setBinaryStream(idx, is, len);
    }

    /**
     * Metoda delegowana do okrelonej instrukcji preparowanej
     * @parametr idx indeks do przypisania wartoci
     * @parametr b warto blob
     * @zglasza java.sql.SQLException wystpi bd bazy danych
     */
    public void setBlob(int idx, Blob b) throws SQLException {
        validate();
        statement.setBlob(idx, b);
    }

    /**
     * Metoda delegowana do okrelonej instrukcji preparowanej
     * @parametr idx indeks do przypisania wartoci
     * @parametr b warto typu boolean
     * @zglasza java.sql.SQLException wystpi bd bazy danych
     */
    public void setBoolean(int idx, boolean b) throws SQLException {
        validate();
        statement.setBoolean(idx, b);
    }

    /**
     * Metoda delegowana do okrelonej instrukcji preparowanej
     * @parametr idx indeks do przypisania wartoci
     * @parametr b warto bajtowa
     * @zglasza java.sql.SQLException wystpi bd bazy danych
     */
    public void setByte(int idx, byte b) throws SQLException {
        validate();
        statement.setByte(idx, b);
    }

    /**
     * Metoda delegowana do okrelonej instrukcji preparowanej
     * @parametr idx indeks do przypisania wartoci
     * @parametr data dane binarne
     * @zglasza java.sql.SQLException wystpi bd bazy danych
     */
    public void setBytes(int idx, byte[] data) throws SQLException {
        validate();
        statement.setBytes(idx, data);
    }

    /**
     * Metoda delegowana do okrelonej instrukcji preparowanej
     * @parametr idx indeks do przypisania wartoci
     * @parametr rdr czytnik, z ktrego naley odczyta warto
     * @parametr len dugo strumienia
     * @zglasza java.sql.SQLException wystpi bd bazy danych
     */
    public void setCharacterStream(int idx, Reader rdr, int len)
        throws SQLException {
        validate();
        statement.setCharacterStream(idx, rdr, len);
    }

    /**
     * Metoda delegowana do okrelonej instrukcji preparowanej.
     * @parametr idx indeks do przypisania wartoci
     * @parametr c warto znakowa duego obiektu
     * @zglasza java.sql.SQLException wystpi bd bazy danych
     */
    public void setClob(int idx, Clob c) throws SQLException {
        validate();
        statement.setClob(idx, c);
    }

    /**
     * Metoda delegowana do okrelonej instrukcji preparowanej.
     * @parametr name nazwa kursora
     * @zglasza java.sql.SQLException wystpi bd bazy danych
     */
    public void setCursorName(String name) throws SQLException {
        validate();
        statement.setCursorName(name);
    }

    /**
     * Metoda delegowana do okrelonej instrukcji preparowanej.
     * @parametr idx indeks do przypisania wartoci
     * @parametr d warto daty
     * @zglasza java.sql.SQLException wystpi bd bazy danych
     */
    public void setDate(int idx, Date d) throws SQLException {
        validate();
        statement.setDate(idx, d);
    }

    /**
     * Metoda delegowana do okrelonej instrukcji preparowanej.
     * @parametr idx indeks do przypisania wartoci
     * @parametr d warto daty
     * @parametr cal kalendarz daty
     * @zglasza java.sql.SQLException wystpi bd bazy danych
     */
    public void setDate(int idx, Date d, Calendar cal) throws SQLException {
        validate();
        statement.setDate(idx, d, cal);
    }

    /**
     * Metoda delegowana do okrelonej instrukcji preparowanej.
     * @parametr idx indeks do przypisania wartoci
     * @parametr d warto typu double
     * @zglasza java.sql.SQLException wystpi bd bazy danych
     */
    public void setDouble(int idx, double d) throws SQLException {
        validate();
        statement.setDouble(idx, d);
    }

    /**
     * Metoda delegowana do okrelonej instrukcji preparowanej.
     * @parametr en czy wcza przetwarzanie znakw sterujcych
     * @zglasza java.sql.SQLException wystpi bd bazy danych
     */
    public void setEscapeProcessing(boolean en) throws SQLException {
        validate();
        statement.setEscapeProcessing(en);
    }

    /**
     * Metoda delegowana do okrelonej instrukcji preparowanej.
     * @parametr dir kierunek pobierania
     * @zglasza java.sql.SQLException wystpi bd bazy danych
     */
    public void setFetchDirection(int dir) throws SQLException {
        validate();
        statement.setFetchDirection(dir);
    }

    /**
     * Metoda delegowana do okrelonej instrukcji preparowanej.
     * @parametr sz rozmiar pobierania
     * @zglasza java.sql.SQLException wystpi bd bazy danych
     */
    public void setFetchSize(int sz) throws SQLException {
        validate();
        statement.setFetchSize(sz);
    }

    /**
     * Metoda delegowana do okrelonej instrukcji preparowanej.
     * @parametr idx indeks do przypisania wartoci
     * @parametr f wartoc typu float
     * @zglasza java.sql.SQLException wystpi bd bazy danych
     */
    public void setFloat(int idx, float f) throws SQLException {
        validate();
        statement.setFloat(idx, f);
    }

    /**
     * Metoda delegowana do okrelonej instrukcji preparowanej.
     * @parametr idx indeks do przypisania wartoci
     * @parametr x warto typu integer
     * @zglasza java.sql.SQLException wystpi bd bazy danych
     */
    public void setInt(int idx, int x) throws SQLException {
        validate();
        statement.setInt(idx, x);
    }

    /**
     * Metoda delegowana do okrelonej instrukcji preparowanej.
     * @parametr idx indeks do przypisania wartoci
     * @parametr l wartoc typu long
     * @zglasza java.sql.SQLException wystpi bd bazy danych
     */
    public void setLong(int idx, long l) throws SQLException {
        validate();
        statement.setLong(idx, l);
    }

    /**
     * Metoda delegowana do okrelonej instrukcji preparowanej.
     * @parametr max maksymalny rozmiar pola
     * @zglasza java.sql.SQLException wystpi bd bazy danych
     */
    public void setMaxFieldSize(int max) throws SQLException {
        validate();
        statement.setMaxFieldSize(max);
    }

    /**
     * Metoda delegowana do okrelonej instrukcji preparowanej.
     * @parametr max maksymalna liczba wierszy
     * @zglasza java.sql.SQLException wystpi bd bazy danych
     */
    public void setMaxRows(int max) throws SQLException {
        validate();
        statement.setMaxRows(max);
    }

    /**
     * Metoda delegowana do okrelonej instrukcji preparowanej.
     * @parametr idx indeks do przypisania wartoci
     * @parametr type typ okrelony w pliku <code>java.sql.Types</code>
     * @zglasza java.sql.SQLException wystpi bd bazy danych
     * @zobacz java.sql.Types
     */
    public void setNull(int idx, int type) throws SQLException {
        validate();
        statement.setNull(idx, type);
    }

    /**
     * Metoda delegowana do okrelonej instrukcji preparowanej.
     * @parametr idx indeks do przypisania wartoci
     * @parametr type typ okreslony w pliku <code>java.sql.Types</code>
     * @parametr name nazwa typu uytkownika
     * @zglasza java.sql.SQLException wystpi bd bazy danych
     * @zobacz java.sql.Types
     */
    public void setNull(int idx, int type, String name) throws SQLException {
        validate();
        statement.setNull(idx, type, name);
    }

    /**
     * Metoda delegowana do okrelonej instrukcji preparowanej.
     * @parametr idx indeks do przypisania wartoci
     * @parametr ob warto typu obiektowego
     * @zglasza java.sql.SQLException wystpi bd bazy danych
     */
    public void setObject(int idx, Object ob) throws SQLException {
        validate();
        statement.setObject(idx, ob);
    }

    /**
     * Metoda delegowana do okrelonej instrukcji preparowanej.
     * @parametr idx indeks do przypisania wartoci
     * @parametr ob warto typu obiektowego
     * @parametr type typ okrelony w pliku <code>java.sql.Types</code>
     * @zglasza java.sql.SQLException wystpi bd bazy danych
     * @zobacz java.sql.Types
     */
    public void setObject(int idx, Object ob, int type) throws SQLException {
        validate();
        statement.setObject(idx, ob, type);
    }

    /**
     * Metoda delegowana do okrelonej instrukcji preparowanej.
     * @parametr idx indeks do przypisania wartoci
     * @parametr ob warto typu obiektowego
     * @parametr type typ okrelony w pliku <code>java.sql.Types</code>
     * @parametr scale skala wartoci
     * @zglasza java.sql.SQLException wystpi bd bazy danych
     * @zobacz java.sql.Types
     */
    public void setObject(int idx, Object ob, int type, int scale)
        throws SQLException {
        validate();
        statement.setObject(idx, ob, type, scale);
    }

    /**
     * Metoda delegowana do okrelonej instrukcji preparowanej.
     * @parametr sec limit czasu zapytania w sekundach
     * @zglasza java.sql.SQLException wystpi bd bazy danych
     */
    public void setQueryTimeout(int sec) throws SQLException {
        validate();
        statement.setQueryTimeout(sec);
    }

    /**
     * Metoda delegowana do okrelonej instrukcji preparowanej.
     * @parametr idx indeks do przypisania wartoci
     * @parametr ref warto odniesienia
     * @zglasza java.sql.SQLException wystpi bd bazy danych
     */
    public void setRef(int idx, Ref ref) throws SQLException {
        validate();
        statement.setRef(idx, ref);
    }

    /**
     * Metoda delegowana do okrelonej instrukcji preparowanej.
     * @parametr idx indeks do przypisania wartoci
     * @parametr s warto typu short
     * @zglasza java.sql.SQLException wystpi bd bazy danych
     */
    public void setShort(int idx, short s) throws SQLException {
        validate();
        statement.setShort(idx, s);
    }

    /**
     * Metoda delegowana do okrelonej instrukcji preparowanej.
     * @parametr idx indeks do przypisania wartoci
     * @parametr str warto typu string
     * @zglasza java.sql.SQLException wystpi bd bazy danych
     */
    public void setString(int idx, String str) throws SQLException {
        validate();
        statement.setString(idx, str);
    }

    /**
     * Metoda delegowana do okrelonej instrukcji preparowanej.
     * @parametr idx indeks do przypisania wartoci
     * @parametr t warto czasu
     * @zglasza java.sql.SQLException wystpi bd bazy danych
     */
    public void setTime(int idx, Time t) throws SQLException {
        validate();
        statement.setTime(idx, t);
    }

     /**
     * Metoda delegowana do okrelonej instrukcji preparowanej.
     * @parametr idx indeks do przypisania wartoci
     * @parametr t warto czasu
     * @parametr cal kalendarz dla wartoci czasu
     * @zglasza java.sql.SQLException wystpi bd bazy danych
     */
   public void setTime(int idx, Time t, Calendar cal) throws SQLException {
        validate();
        statement.setTime(idx, t, cal);
    }

    /**
     * Metoda delegowana do okrelonej instrukcji preparowanej.
     * @parametr idx indeks do przypisania wartoci
     * @parametr ts warto datownika
     * @zglasza java.sql.SQLException wystpi bd bazy danych
     */
    public void setTimestamp(int idx, Timestamp ts) throws SQLException {
        validate();
        statement.setTimestamp(idx, ts);
    }

    /**
     * Metoda delegowana do okrelonej instrukcji preparowanej.
     * @parametr idx indeks do przypisania wartoci
     * @parametr ts wartoc datownika
     * @parametr cal kalendarz dla wartoci datownika
     * @zglasza java.sql.SQLException wystpi bd bazy danych
     */
    public void setTimestamp(int idx, Timestamp ts, Calendar cal)
        throws SQLException {
        validate();
        statement.setTimestamp(idx, ts, cal);
    }

    /**
     * Metoda delegowana do okrelonej instrukcji preparowanej.
     * @parametr idx indeks do przypisania wartoci
     * @parametr is the strumie wejciowy
     * @parametr len dugo strumienia wejciowego
     * @zglasza java.sql.SQLException wystpi bd bazy danych
     * @deprecated use setCharacterStream()
     */
    public void setUnicodeStream(int idx, InputStream is, int len)
        throws SQLException {
        validate();
        statement.setUnicodeStream(idx, is, len);
    }

    /**
     * Sprawdza, czy ta instrukcja jest jeszcze wana.
     * @zglasza java.sql.SQLException instrukcja nie jest ju wana
     */
    private void validate() throws SQLException {
        if( isClosed() ) {
            throw new SQLException("Niedozwolona prba dostpu do instrukcji.");
        }
    }

    public String toString() {
        if( statement != null ) {
            return super.toString() + " [" + statement.toString() + "]";
        }
        else {
            return super.toString();
        }
    }
}
