/* $Id$ */
/* Copyright R 2002 George Reese, Imaginet */
package org.dasein.persist;

// Opracowa  George Reese dla potrzeb ksiki:
// Java. Aplikacje bazodanowe. Najlepsze rozwizania: J2EE
// Przenis na bibliotek kodu digital@jwt George Reese


import java.sql.CallableStatement;
import java.sql.Connection;
import java.sql.DatabaseMetaData;
import java.sql.PreparedStatement;
import java.sql.ResultSet;
import java.sql.Statement;
import java.sql.SQLException;
import java.sql.SQLWarning;
import java.util.ArrayList;
import java.util.Iterator;
import java.util.Map;

import javax.naming.InitialContext;
import javax.naming.NamingException;
import javax.sql.ConnectionEvent;
import javax.sql.ConnectionEventListener;
import javax.sql.DataSource;
import javax.sql.PooledConnection;

/**
 * Abstrakcja poczenia fizycznego dla poczenia JDBC
 * producenta bazy danych. Ta abstrakcja istnieje w puli
 * pocze i realizuje tworzenie puli instrukcji.<br/>
 * Ostatnio zmodyfiokowano $Date$
 * @wersja $Revision$
 * @autor George Reese
 */
public class PhysicalConnection
implements PooledConnection, Connection, StatementEventListener {
    /**
     * Rzeczywiste poczenie JDBC z baz danych. Klasa ta dziaa
     * z dowolnym egzemplarzem poczenia zgodnym z JDBC. W celu zapewnienia
     * najlepszej wydajnoci przekazane poczenie nie powinno by umieszczane w puli.
     */
    private Connection        database  = null;
    /**
     * Lista obiektw nasuchujcych zdarzenia zwizane z poczeniem.
     */
    private ArrayList         listeners = new ArrayList();
    /**
     * Aktualnie otwarte poczenie logiczne.
     */
    private LogicalConnection logical   = null;
    /**
     * Pula instrukcji otwarta przez to poczenie.
     */
    private StatementPool     pool      = new StatementPool();

    /**
     * Tworzy nowe poczenie fizyczne pobierajce zwizane z nim
     * poczenie z baz danych z okrelonego rda danych.
     * @parametr dsn nazwa rda danych zapewniajcego poczenia JDBC      * @zglasza java.sql.SQLException w przypadku wystpienia bdu      * poczenia z baz danych
     */
    public PhysicalConnection(String dsn) throws SQLException {
        this(dsn, null, null);
    }

    /**
     * Tworzy nowe poczenie fizyczne pobierajce zwizane z nim
     * poczenie z baz danych z okrelonego rda danych.
     * @parametr dsn nazwa rda danych zapewniajcego poczenia JDBC
     * @parametr uid identyfikator uytkownika do nawizania poczenia
     * @parametr pw  haso do nawizania poczenia
     * @zglasza java.sql.SQLException w przypadku bdu poczenia z     * baz danych
     */
    public PhysicalConnection(String dsn, String uid, String pw)
        throws SQLException {
        super();
        try {
            InitialContext ctx = new InitialContext();
            DataSource ds = (DataSource)ctx.lookup(dsn);

            if( uid == null && pw == null ) {
                database = ds.getConnection();
            }
            else {
                database = ds.getConnection(uid, pw);
            }
        }
        catch( NamingException e ) {
            throw new SQLException(e.getMessage());
        }
        Thread t = new Thread() {
               public void run() {
                    keepAlive();
                }
            };

        t.setDaemon(true);
        t.setName("KEEP ALIVE");
        t.start();
    }

    /**
     * Dodaje nowy proces nasuchujcy zdarze zwizanych z poczeniem. Metoda ta jest
     * czci "kontraktu" pocze w puli ze rodowiskiem wykorzystujcym pul      * polegajcym na tym, e rodowisko "wie" kiedy nastpuje poczenie     * oraz nieodwracalny bd.
     * @parametr lstnr nowy proces nasuchujcy
     */
    public void addConnectionEventListener(ConnectionEventListener lstnr) {
        synchronized( listeners ) {
            listeners.add(lstnr);
        }
    }

    /**
     * Metoda delegowana do okrelonego poczenia JDBC.
     * @zglasza java.sql.SQLException wystpi bd bazy danych
     */
    public void clearWarnings() throws SQLException {
        database.clearWarnings();
    }

    /**
     * Metoda delegowana do okrelonego poczenia JDBC.
     * @zglasza java.sql.SQLException wystpi bd bazy danych
     */
    public void close() throws SQLException {
        synchronized( listeners ) {
            if( database != null ) {
                database.close();
            }
        }
        logical = null;
    }

    /**
     * Metoda wywoywania przez biece poczenie logiczne w celu poinformowania
     * poczenia fizycznego o zamkniciu poczenia logicznego.
     * Metoda ta powoduje zatem zainicjowanie zdarzenia zwizanego z poczeniem,     * ktre informuje rodowisko puli, e to poczenie fizyczne      * jest dostpne do wielokrotnego wykorzystania.
     */
    public void connectionClosed() {
        synchronized( listeners ) {
            Iterator it = listeners.iterator();
            
            while( it.hasNext() ) {
                ConnectionEventListener lstnr;

                lstnr = (ConnectionEventListener)it.next();
                lstnr.connectionClosed(new ConnectionEvent(this));
            }
            logical = null;
            listeners.notifyAll();
        }
    }

    /**
     * Metoda wywoywania przez biece poczenie logiczne w celu poinformowania
     * poczenia fizycznego o wystpieniu bdu, z powodu ktrego wystpuje konieczno
     * odrzucenia tego poczenia. Metoda ta powoduje zatem zainicjowanie zdarzenia
     * zwizanego  z poczeniem, ktre informuje rodowisko puli, e to poczenie
     * fizyczne jest na trwae niedostpne do wykorzystania.
     * @parametr e wyjtek bdcy przyczyn odrzucenia poczenia
     */
    public void connectionErrored(SQLException e) {
        synchronized( listeners ) {
            Iterator it = listeners.iterator();

            while( it.hasNext() ) {
                ConnectionEventListener lstnr;

                lstnr = (ConnectionEventListener)it.next();
                lstnr.connectionErrorOccurred(new ConnectionEvent(this, e));
            }
        }
    }

    /**
     * Metoda delegowana do poczenia JDBC.
     * @zglasza java.sql.SQLException w przypadku wystpienia bedu bazy danych
     */
    public void commit() throws SQLException {
        database.commit();
    }

    /**
     * Metoda delegowana do okrelonego poczenia JDBC.
     * @zwraca nowy egzemplarz instrukcji
     * @zglasza java.sql.SQLException wystpi bd bazy danych
     */
    public Statement createStatement() throws SQLException {
        return database.createStatement();
    }

    /**
     * Metoda delegowana do okrelonego poczenia JDBC.
     * @parametr rst typ zestawu wynikw
     * @parametr rsc wspbiecno zestawu wynikw
     * @zglasza java.sql.SQLException wystpi bd bazy danych
     */
    public Statement createStatement(int rst, int rsc) throws SQLException {
        return database.createStatement(rst, rsc);
    }

    /**
     * Metoda delegowana do poczenia JDBC.
     * @zwraca biecy stan autozatwierdzania
     * @zglasza java.sql.SQLException wystpi bd bazy danych
     */
    public boolean getAutoCommit() throws SQLException {
        return database.getAutoCommit();
    }

    /**
     * Metoda delegowana do okrelonego poczenia JDBC.
     * @zwraca aktualnie wybrany katalog
     * @zglasza java.sql.SQLException wystpi bd bazy danych
     */
    public String getCatalog() throws SQLException {
        return database.getCatalog();
    }

    /**
     * Dostarcza do rodowiska puli poczenie logiczne
     * odwoujce si do poczenia fizycznego.
     * @zwraca poczenie logiczne do wykorzystania przez aplikacj
     * @zgasza java.sql.SQLException wystpi bd bazy danych
     */
    public Connection getConnection() throws SQLException {
        synchronized( listeners ) {
            logical = new LogicalConnection(this);
            return logical;
        }
    }

    /**
     * Metoda delegowana do okrelonego poczenia JDBC.
     * @zwraca metadane dla tego poczenia
     * @zglasza java.sql.SQLException wystpi bd bazy danych
     */
    public DatabaseMetaData getMetaData() throws SQLException {
        return database.getMetaData();
    }

    /**
     * Metoda delegowana do okrelonego poczenia JDBC.
     * @zwraca biecy poziom izolacji transackji
     * @zglasza java.sql.SQLException wystpi bd bazy danych
     */
    public int getTransactionIsolation() throws SQLException {
        return database.getTransactionIsolation();
    }

    /**
     * Metoda delegowana do okrelonego poczenia JDBC.
     * @zwraca biec map typw
     * @zglasza java.sql.SQLException wystpi bd bazy danych
     */
    public Map getTypeMap() throws SQLException {
        return database.getTypeMap();
    }

    /**
     * Metoda delegowana do poczenia JDBC.
     * @zwraca ostrzeenia
     * @zglasza java.sql.SQLException wystpi bd bazy danych
     */
    public SQLWarning getWarnings() throws SQLException {
        return database.getWarnings();
    }

    /**
     * Metoda delegowana do poczenia JDBC.
     * @zwraca true jeeli nie ma poczenia, albo jeeli      * to poczenie zamknito
     * @zglasza java.sql.SQLException wystpi bd bazy danych
     */
    public boolean isClosed() throws SQLException {
        return (database != null && database.isClosed());
    }

    /**
     * Metoda delegowana do okrelonego poczenia JDBC.
     * @zwraca biecy status tylko do odczytu
     * @zglasza java.sql.SQLException wystpi bd bazy danych
     */
    public boolean isReadOnly() throws SQLException {
        return database.isReadOnly();
    }

    /**
     * Ta metoda jest wywoywana w ptli. Jej zadanie jest podtrzymywanie poczenia
     * znajdujcego si w puli. Metoda realizuje t funkcj poprzez
     * okresowo wysyany kod:<br/>
     * <code>SELECT 1</code>
     * do aktywnej bazy danych.
     * @zwraca ostrzeenia
     * @zglasza java.sql.SQLException wystpi bd bazy danych
     */
    private void keepAlive() {
        synchronized( listeners ) {
            boolean closed = false;

            do {
                try { listeners.wait(60000); }
                catch( InterruptedException e ) { }
                // operacja wykonywana tylko wtedy, gdy poczenie znajduje si w puli
                if( logical == null ) {
                    // jeeli wystpi bd podczas sprawdzania
                    // poczenie zostanie usunite z puli
                    if( !test() ) {
                        try { database.close(); }
                        catch( SQLException e ) { }
                        database = null;
                        logical = null;
                        return;
                    }
                }
                try {
                    closed = isClosed();
                }
                catch( SQLException e ) {
                    closed = true;
                }
            } while( !closed );
        }
    }

    /**
     * Metoda delegowana do okrelonego poczenia JDBC.
     * @parametr sql instrukcja ANSI SQL do przeksztacenia
     * @zwraca rodzimy kod SQL dla okrelonej instrukcji ANSI SQL
     * @zglaszas java.sql.SQLException wystpi bd bazy danych
     */
    public String nativeSQL(String sql) throws SQLException {
        return database.nativeSQL(sql);
    }

    /**
     * Metoda delegowana do okrelonego poczenia JDBC.
     * @parametr sql procedura skadowana do wywoania
     * @zwraca instrukcj wywoywaln dla okreslonej procedury skadowanej
     * @zglasza java.sql.SQLException wystpi bd bazy danych.
     */
    public CallableStatement prepareCall(String sql) throws SQLException {
        return database.prepareCall(sql);
    }

    /**
     * Metoda delegowana do okrelonego poczenia JDBC.
     * @parametr sql procedura skadowana do wywoania
     * @parametr rst typ zestawu wynikw
     * @parametr rsc wspbieno zestawu wynikw
     * @zwraca instrukcj wywoywaln dla okrelonej procedury skadowanej
     * @zglasza java.sql.SQLException wystpi bd bazy danych
     */
    public CallableStatement prepareCall(String sql, int rst, int rsc)
        throws SQLException {
        return database.prepareCall(sql, rst, rsc);
    }

    /**
     * Przeszukuje pul instrukcji preparowanych w celu odnalezienia pasujcego SQL.
     * W przypadku niepowodzenia utworzenie nowej instrukcji.
     * @parametr sql preparowana instrukcja SQL
     * @zwraca instrukcj preparowan z puli
     * @zglasza java.sql.SQLException w przypadku beduu bazy danych
     */
    public PreparedStatement prepareStatement(String sql)
        throws SQLException {
        PreparedStatement stmt;
        PooledStatement ps;

        synchronized( pool ) {
            if( pool.contains(sql) ) {
                stmt = pool.pop(sql);
            }
            else {
                stmt = database.prepareStatement(sql);
            }
        }
        ps = new PooledStatement(logical, sql, stmt);
        ps.addStatementEventListener(this);
        return ps;
    }

    /**
     * Przygotowuje instrukcj dla okrelonego kodu SQL. Jeeli pula instrukcji
     * zawiera okrelon instrukcj, a spodziewanym zestawem wynikw jest     * <code>ResultSet.TYPE_FORWARD_ONLY</code> oraz
     * <code>ResultSet.CONCUR_READ_ONLY</code>, wwczas instrukcja jest
     * pobierana z puli. Dla wszystkich pozostaych instrukcji
     * przygotowywana jest nowa instrukcja. W tym mechanizmie puli
     * nie s obsugiwane zestawy wynikw z moliwoci przewijania i aktualizacji.
     * @parametr sql instrukcja SQL do spreparowania
     * @parametr rst typ zestawu wynikw
     * @parametr rsc wspbieno zestawu wynikw
     * @zwraca instrukcj  preparowan dla okreslonej instrukcji SQL
     * @zglasza java.sql.SQLException wystpi bd bazy danych
     */
    public PreparedStatement prepareStatement(String sql, int rst, int rsc)
        throws SQLException {
        if( rst == ResultSet.TYPE_FORWARD_ONLY ) {
            if( rsc == ResultSet.CONCUR_READ_ONLY ) {
                return prepareStatement(sql);
            }
        }
        return database.prepareStatement(sql);
    }

    /**
     * Usuwa proces nasuchujcy z listy zgodnie z     * "kontraktem" poczenia z puli ze rodowiskiem puli.      *
     * @param lstnr the proces nasuchujcy do usunicia
     */
    public void removeConnectionEventListener(ConnectionEventListener lstnr) {
        synchronized( listeners ) {
            listeners.remove(lstnr);
        }
    }

    /**
     * Metoda delegowana do okrelonego poczenia JDBC.
     * @zglasza java.sql.SQLException wystpi bd bazy danych
     */
    public void rollback() throws SQLException {
        database.rollback();
    }

    /**
     * Metoda delegowana do okrelonego poczenia JDBC.
     * @parametr ac status autozatwierdzania do przypisania
     * @zglasza java.sql.SQLException wystpi bd bazy danych
     */
    public void setAutoCommit(boolean ac) throws SQLException {
        database.setAutoCommit(ac);
    }

    /**
     * Metoda delegowana do okrelonego poczenia JDBC.
     * @parametr cat katalog do przypisania
     * @zglasza java.sql.SQLException wystpi bd bazy danych
     */
    public void setCatalog(String cat) throws SQLException {
        database.setCatalog(cat);
    }

    /**
     * Metoda delegowana do okrelonego poczenia JDBC.
     * @parametr ro status tylko do odczytu do przypisania
     * @zglasza java.sql.SQLException wystpi bd bazy danych
     */
    public void setReadOnly(boolean ro) throws SQLException {
        database.setReadOnly(ro);
    }

    /**
     * Metoda delegowana do okrelonego poczenia JDBC.
     * @param lvl poziom izolacji transakcji do przypisania
     * @zglaza java.sql.SQLException wystpi bd bazy danych
     */
    public void setTransactionIsolation(int lvl) throws SQLException {
        database.setTransactionIsolation(lvl);
    }

    /**
     * Metoda delegowana do okrelonego poczenia JDBC.
     * @parametr map mapa typw do przypisania
     * @zglaszas java.sql.SQLException wystpi bd bazy danych
     */
    public void setTypeMap(Map map) throws SQLException {
        database.setTypeMap(map);
    }

    /**
     * Wywoywana przez instrukcj w przypadku zamknicia jej przez aplikacj.
     * Oznacza to moliwo zwrcenia instrukcji do puli.
     * @parametr evt zdarzenie powodujca zwrot instrukcji do puli
     * @zglasza java.sql.SQLException wystpi bd bazy danych
     */
    public void statementClosed(StatementEvent evt) {
        synchronized( pool ) {
            pool.push(evt.getSQL(), evt.getStatement());
        }
    }

    /**
     * Testuje poczenie w celu sprawdzenia, czy jest aktywne. Jeeli nie, powiadamia
     * rodowisko puli, e to poczenie nie jest ju aktywne.
     */
    private boolean test() {
        Statement stmt = null;

        try {
            stmt = database.createStatement();
            stmt.executeQuery("SELECT 1");
            return true;
        }
        catch( SQLException e ) {
            connectionErrored(e);
            return false;
        }
        finally {
            if( stmt != null ) {
                try { stmt.close(); }
                catch( SQLException e ) { }
            }
        }
    }

    public String toString() {
        if( database != null ) {
            return super.toString() + " [" + database.toString() + "]";
        }
        else {
            return super.toString();
        }
    }
}
