/* $Id$ */
/* Copyright R 2002 George Reese, Imaginet */
package org.dasein.persist;

// Opracowa  George Reese dla potrzeb ksiki:
// Java. Aplikacje bazodanowe. Najlepsze rozwizania: J2EE
// Przenis na bibliotek kodu digital@jwt George Reese

import java.io.PrintWriter;
import java.sql.SQLException;

import javax.sql.ConnectionPoolDataSource;
import javax.sql.PooledConnection;

/**
 * Implementacja rda danych puli pocze umoliwiajcej
 * umieszczanie w puli instrukcji preparowanych. To rdo danych zaley od
 * innego rda w celu zapewnienia waciwego poczenia z baz danych.
 * To drugie rdo nie powinno by umieszczane w puli.<br/>
 * Data ostatniej modyfikacji $Date$
 * @wersja $Revision$
 * @autor George Reese
 */
public class ConnectionPool implements ConnectionPoolDataSource {
    /**
     * Wewntrzna klasa dokonujca prby poczenia z baz danych.
     */
    static private class ConnectionAttempt extends Thread {
        /**
         * Odwoanie do poczenia w puli reprezentujce sukces.
         */
        private PhysicalConnection connection = null;
        /**
         * Nazwa rda danych udostpniajcego poczenie z baz danych.
         */
        private String             dsn        = null;
        /**
         * Wyjtki SQL zgaszane podczas poczenia.
         */
        private SQLException       exception  = null;
        /**
         * haso wykorzystane dla poczenia.
         */
        private String             password   = null;
        /**
         * Identyfikator uytkownika do nawizania poczenia.
         */
        private String             userID     = null;
        
        /**
         * Konstrukcja nowego egzemplarza.
         * @parametr dsn nazwa rda danych dla waciwego poczenia
         * @parametr uid identyfikator uytkownika do wykorzystania w celu nawizania poczenia
         * @parametr pw haso uyte do nawizania poczenia
         */
        public ConnectionAttempt(String dsn, String uid, String pw) {
            super();
            this.dsn = dsn;
            this.userID = uid;
            this.password = pw;
            start();
        }

        /**
         * Nawizanie poczenia i ustawienie atrybutu <code>connection</code>
         * lub <code>exception</code>. Po zakoczeniu, wywoanie
         * <code>this.notifyAll()</code>.
         */
        public void run() {
            synchronized( this ) {
                try {
                    connection = new PhysicalConnection(dsn, userID, password);
                }
                catch( SQLException e ) {
                    exception = e;
                }
                notifyAll();
            }
        }
    }

    /**
     * Nazwa rda danych wykorzystanego do nawizania waciwego
     * poczenia z baz danych. Podczas konfiguracji tej puli poczenia
     * w repozytorium JNDI, naley okreli warto dla tej
     * waciwoci.
     */
    private String      dsn          = null;
    /**
     * Wymagana nazwa procedury zapisujcej dzienniki zdarze.
     */
    private PrintWriter logWriter    = null;
    /**
     * wymagany parametr limitu czasu logowania.
     */
    private int         loginTimeout = 0;

    /**
     * Nazwa rda danych, ktre ta pula wykorzystuje
     * do nawizywania waciwych pocze.
     * @zwraca nazw rda danych
     */
    public String getDsn() {
        return dsn;
    }

    /**
     * @zwraca nazw procedury zapisujcej dziennik zdarze
     * @zglasza java.sql.SQLException w rzeczywistoci nigdy nie zgaszany
     */
    public PrintWriter getLogWriter() throws SQLException {
        return logWriter;
    }

    /**
     * @zwraca limit czasu logowania
     * @zglasza java.sql.SQLException w rzeczywistoci nigdy nie zgaszany
     */
    public int getLoginTimeout() throws SQLException {
        return loginTimeout;
    }

    /**
     * Udostpnia aplikacji rdo danych w powizaniu z
     * poczeniem z puli. Poczenie z puli reprezentuje
     * fizyczne poczenie z baz danych. Poniewa ta biblioteka jest
     * abstrakcj majc na celu zaprezentowanie puli instrukcji,
     * nasze fizyczne poczenie take jest abstrakcj. Zadaniem
     * poczenia fizycznego jest udostpnienie aplikacji
     * rda danych z logicznym poczeniem.
     * @zwraca poczenie z baz danych w puli
     * @zglasza java.sql.SQLException w przypadku bdu poczenia
     */
    public PooledConnection getPooledConnection() throws SQLException {
        return getPooledConnection(null, null);
    }

    /**
     * Udostpnia aplikacji rdo danych w odniesieniu do
     * poczenia z puli na podstawie okrelonych danych uwierzytelniania.
     * @parametr uid Id uytkownika do wykorzystania podczas nawizywania poczenia
     * @param pw haso do zastosowania podczas nawizywania poczenia
     * @zwraca poczenie w puli
     * @zglasza java.sql.SQLException w przypadku bdu poczenia
     */
    public PooledConnection getPooledConnection(String uid, String pw)
        throws SQLException {
        if( loginTimeout > 0 ) {
            ConnectionAttempt att = new ConnectionAttempt(dsn, uid, pw);
            long start = System.currentTimeMillis();
            long to = (long)(loginTimeout * 1000);

            synchronized( att ) {
                while( (System.currentTimeMillis() - start) < to ) {
                    try { att.wait(to); }
                    catch( InterruptedException e ) { }
                    if( att.connection != null ) {
                        return att.connection;
                    }
                    else if( att.exception != null ) {
                        throw att.exception;
                    }
                    else {
                        to = to - (System.currentTimeMillis() - start);
                    }
                }
                throw new SQLException("Przekroczony limit czasu poczenia.");
            }
        }
        else {
            return new PhysicalConnection(dsn, uid, pw);
        }
    }

    /**
     * Ustawia nazw rda danych do wykorzystania w celu generowania pocze.
     * @parametr dsn nazwa rda danych
     */
    public void setDsn(String dsn) {
        this.dsn = dsn;
    }

    /**
     * Przypisuje procedur zapisujc dane o zdarzeniach w dzienniku.
     * @parametr out nowa procedura zapisujca
     * @zglasza java.sql.SQLException - nigdy nie zgaszany
     */
    public void setLogWriter(PrintWriter out) throws SQLException {
        logWriter = out;
    }

    /**
     * Ustawia nowy limit czasu poczenia dla tej puli.
     * @parametr sec limit czasu logowania w sekundach
     * @zglasza java.sql.SQLException - nigdy nie zgaszany
     */
    public void setLoginTimeout(int sec) throws SQLException {
        loginTimeout = sec;
    }
}
