/* $Id$ */
/* Copyright  2002 George Reese, Imaginet */
package org.dasein.persist;

// Opracowa  George Reese dla potrzeb ksiki:
// Java. Aplikacje bazodanowe. Najlepsze rozwizania: J2EE
// Przenis na bibliotek kodu digital@jwt George Reese

import java.sql.CallableStatement;
import java.sql.Connection;
import java.sql.DatabaseMetaData;
import java.sql.PreparedStatement;
import java.sql.Statement;
import java.sql.SQLException;
import java.sql.SQLWarning;
import java.util.ArrayList;
import java.util.Iterator;
import java.util.Map;

/**
 * Reprezentuje logiczne poczenie z baz danych do wykorzystania przez
 * aplikacj. W przypadku zamknicia, to logiczne poczenie zwraca
 * poaczenie fizyczne z powrotem do puli pocze.<br/>
 * Data ostatniej modyfikacji $Date$
 * @wersja $Revision$
 * @autor George Reese
 */
public class LogicalConnection implements Connection {
    /**
     * Poczenie fizyczne powizane z tym poaczeniem logicznym.
     */
    private PhysicalConnection physical  = null;
    /**
     * Instrukcje wygenerowane przez to poczenie.
     */
    private ArrayList         statements = new ArrayList();

    /**
     * Tworzy nowe poczenie logiczne powizane z okrelonym
     * poczeniem fizycznym.
     */
    public LogicalConnection(PhysicalConnection conn) {
        super();
        physical = conn;
    }

    /**
     * Deleguje do poczenia fizycznego.
     * @zglasza java.sql.SQLException w przypadku bdu bazy danych
     */
    public void clearWarnings() throws SQLException {
        validate();
        physical.clearWarnings();
    }

    /**
     * Informuje poczenie fizyczne, e to poczenie logiczne
     * nie jest ju otwarte, dziki czemu poaczenie fizyczne mona zwrci
     * do puli pocze. Po wywoaniu tej metody, nastpne prby
     * aplikacji uycia tego poczenia logicznego
     * zakocz si zgoszeniem wyjtku.
     * @zglasza java.sql.SQLException w przypadku bdu bazy danych
     */
    public void close() throws SQLException {
        Iterator it = statements.iterator();
        
        validate();
        while( it.hasNext() ) {
            PreparedStatement stmt = (PreparedStatement)it.next();

            try { stmt.close(); }
            catch( SQLException e ) { }
        }
        statements.clear();
        physical.connectionClosed();
        physical = null;
    }

    /**
     * Deleguje do poczenia fizycznego.
     * @zglasza java.sql.SQLException w przypadku bdu bazy danych
     */     
    public void commit() throws SQLException {
        validate();
        physical.commit();
    }

    /**
     * Deleguje do poczenia fizycznego.
     * @zwraca nowy egzemplarz instrukcji
     * @zglasza java.sql.SQLException w przypadku bdu bazy danych
     */
    public Statement createStatement() throws SQLException {
        validate();
        return physical.createStatement();
    }

    /**
     * Deleguje do poczenia fizycznego.
     * @parametr rst typ zestawu wynikw
     * @parametr rsc wspbieny zestaw wynikw
     * @zwraca  nowy egzemplarz instrukcji
     * @zglasza java.sql.SQLException w przypadku bdu bazy danych
     */
    public Statement createStatement(int rst, int rsc) throws SQLException {
        validate();
        return physical.createStatement(rst, rsc);
    }

    /**
     * Deleguje do poczenia fizycznego.
     * @zwraca  biecy status autozatwierdzania
     * @zglasza java.sql.SQLException w przypadku bdu bazy danych
     */    
    public boolean getAutoCommit() throws SQLException {
        validate();
        return physical.getAutoCommit();
    }

    /**
     * Deleguje do poczenia fizycznego.
     * @zwraca  ostatnio wybrany katalog
     * @zglasza java.sql.SQLException w przypadku bdu bazy danych     
     */
    public String getCatalog() throws SQLException {
        validate();
        return physical.getCatalog();
    }
    
    /**
     * Deleguje do poczenia fizycznego.
     * @zwraca  metadane dla biecej bazy danych
     * @zglasza java.sql.SQLException w przypadku bdu bazy danych
     */
    public DatabaseMetaData getMetaData() throws SQLException {
        validate();
        return physical.getMetaData();
    }

    /**
     * Deleguje do poczenia fizycznego.
     * @zwraca  biecy poziom izolacji transakcji
     * @zglasza java.sql.SQLException w przypadku bdu bazy danych
     */
    public int getTransactionIsolation() throws SQLException {
        validate();
        return physical.getTransactionIsolation();
    }

    /**
     * Deleguje do poczenia fizycznego.
     * @zwraca  biece odwzorowanie typu
     * @zglasza java.sql.SQLException w przypadku bdu bazy danych
     */
    public Map getTypeMap() throws SQLException {
        validate();
        return physical.getTypeMap();
    }
    
    /**
     * Deleguje do poczenia fizycznego.
     * @zwraca  ostrzeenia
     * @zglasza java.sql.SQLException w przypadku bdu bazy danych
     */
    public SQLWarning getWarnings() throws SQLException {
        validate();
        return physical.getWarnings();
    }
    
    /**
     * Ta metoda sprawdza, czy istnieje poprawne poczenie
     * fizyczne.
     * @zwraca  true jeeli poczenie fizyczne nie jest ju dostpne
     * @zglasza java.sql.SQLException nigdy nie zgasza
     */
    public boolean isClosed() throws SQLException {
        return (physical == null);
    }

    /**
     * Deleguje do poczenia fizycznego.
     * @zwraca  informacj o tym, czy wybrane poczenie jest tylko do odczytu
     * @zglasza java.sql.SQLException w przypadku bdu bazy danych
     */
    public boolean isReadOnly() throws SQLException {
        validate();
        return physical.isReadOnly();
    }
    
    /**
     * Deleguje do poczenia fizycznego.
     * @zwraca  rodzimy kod SQL do wybranej bazy danych dla okrelonego
     * standardu ANSI SQL
     * @zglasza java.sql.SQLException w przypadku bdu bazy danych
     */
    public String nativeSQL(String sql) throws SQLException {
        validate();
        return physical.nativeSQL(sql);
    }
    
    /**
     * Deleguje do poczenia fizycznego.
     * @parametr nazwa procedury skadowanej
     * @zwraca  wywoywaln instrukcj dla wybranej procedury skadowanej
     * @zglasza java.sql.SQLException w przypadku bdu bazy danych
     */
    public CallableStatement prepareCall(String sql) throws SQLException {
        validate();
        return physical.prepareCall(sql);
    }

    /**
     * Deleguje do poczenia fizycznego.
     * @parametr nazwa procedury skadowanej
     * @parametr rst typ zestawu wynikw
     * @parametr wspbieno zestawu wynikw
     * @zwraca  wywoywaln instrukcj dla wybranej procedury skadowanej
     * @zglasza java.sql.SQLException w przypadku bdu bazy danych
     */
    public CallableStatement prepareCall(String sql, int rst, int rsc)
        throws SQLException {
        validate();
        return physical.prepareCall(sql, rst, rsc);
    }
    
    /**
     * Deleguje do poczenia fizycznego.
     * @parametr preparowany SQL
     * @zwraca  instrukcj preparowan dla okrelonego kodu SQL
     * @zglasza java.sql.SQLException w przypadku bdu bazy danych
     */
    public PreparedStatement prepareStatement(String sql)
        throws SQLException {
        PreparedStatement stmt;
        
        validate();
        stmt = physical.prepareStatement(sql);
        statements.add(stmt);
        return stmt;
    }

    /**
     * Deleguje do poczenia fizycznego.
     * @parametr preparowany SQL
     * @parametr rst typ zestawu wynikw
     * @parametr wspbieno zestawu wynikw
     * @zwraca  instrukcj preparowan dla okrelonego kodu SQL
     * @zglasza java.sql.SQLException w przypadku bdu bazy danych
     */
    public PreparedStatement prepareStatement(String sql, int rst, int rsc)
        throws SQLException {
        PreparedStatement stmt;
        
        validate();
        stmt = physical.prepareStatement(sql, rst, rsc);
        statements.add(stmt);
        return stmt;
    }

    /**
     * Deleguje do poczenia fizycznego.
     * @zglasza java.sql.SQLException w przypadku bdu bazy danych
     */
    public void rollback() throws SQLException {
        validate();
        physical.rollback();
    }
    
    /**

     * Deleguje do poczenia fizycznego.
     * @parametr ac status autozatwierdzania do przypisania
     * @zglasza java.sql.SQLException w przypadku bdu bazy danych
     */
    public void setAutoCommit(boolean ac) throws SQLException {
        validate();
        physical.setAutoCommit(ac);
    }

    /**
     * Deleguje do poczenia fizycznego.
     * @parametr cat status katalogu do wyboru
     * @zglasza java.sql.SQLException w przypadku bdu bazy danych
     */
    public void setCatalog(String cat) throws SQLException {
        validate();
        physical.setCatalog(cat);
    }
    
    /**
     * Deleguje do poczenia fizycznego.
     * @parametr ro status tylko do odczytu
     * @zglasza java.sql.SQLException w przypadku bdu bazy danych
     */
    public void setReadOnly(boolean ro) throws SQLException {
        validate();
        physical.setReadOnly(ro);
    }

    /**
     * Deleguje do poczenia fizycznego.
     * @parametr lvl poziom izolacji transakcji do przypisania
     * @zglasza java.sql.SQLException w przypadku bdu bazy danych
     */
    public void setTransactionIsolation(int lvl) throws SQLException {
        validate();
        physical.setTransactionIsolation(lvl);
    }

    /**
     * Deleguje do poczenia fizycznego.
     * @parametr map plik odwzorowa do zastosowania dla odwzorowania typw
     * @zglasza java.sql.SQLException w przypadku bdu bazy danych
     */
    public void setTypeMap(Map map) throws SQLException {
        validate();
        physical.setTypeMap(map);
    }

    public String toString() {
        if( physical != null ) {
            return super.toString() + ("Fizyczne: " + physical.toString() +")";
        }
        else {
            return super.toString();
        }
    }
    
    private void validate() throws SQLException {
       if( isClosed() ) {
            throw new SQLException("Niedozwolona prba wielokrotnego wykorzystania poczenia.");
        }
    }
}
