package s2wexample.controller;

import javax.servlet.*;
import javax.servlet.http.*;
import java.io.*;
import javax.xml.parsers.*;
import org.w3c.dom.*;

/**
 * Implementacja interfejsu umoliwiajcego wykonywanie
 * cieek zada definiowanych jako sekwencje stanw 
 * i posiadajcych akcje pozwalajce wykrywa przejcia pomidzy stanami.
 */
public class WorkflowDispatcher implements Dispatcher {
    /** etykiety uywane przez definicje cieek zada i stanw w jzyku XML*/
    private static final String WORKFLOW_TAG = "workflow";
    private static final String STATE_TAG = "state";
    
    /** atrybuty wsplne ze stronami JSP */
    private static final String NAME_ATTR = "name";
    private static final String ACTION_ATTR = "action";
    private static final String VIEW_ATTR = "viewURI";
    
    /** lokalizacja klas reprezentujcych akcje */
    private static final String ACTION_PREFIX = "s2wexample.controller.actions.";
    
    /**
     * Wewntrzna reprezentacja stanw opisanych w pliku XML
     */
    class State {
        /** nazwa stanu */
        protected String name;
        
        /** akcja zwizana z przejciem do stanu */
        protected Action action;
        
        /** adres uri, ktry naley wywietli */
        protected String viewUri;
    }
    
    /** kontekst serwletu */
    private ServletContext context;
    
    /** tablica stanw */
    private State[] states;
    
    /** biecy stan */
    private int currentState;
    
    /**
     * Wywoywana przez kontener w celu okrelenia kontekstu serwletu
     * podczas inicjacji.
     */
    public void setContext(ServletContext context) throws IOException {
        // przechowuje kontekst
        this.context = context;
        
        // aduje opis cieki zada z pliku i parsuje go
        InputStream is = context.getResourceAsStream("/LanguageWorkflow.xml");
        try {
            states = parseXML(is);
        } catch(Exception ex) {
            throw new IOException(ex.getMessage());
        }
        
        // inicjuje biecy stan
        currentState = 0;
    }
    
    /**
     * Okrela nastepn stron, do ktrej naley przej
     */
    public String getNextPage(HttpServletRequest req) {
        State s = states[currentState];
        
        // zmienia stan tylko wtedy, gdy wykonanie akcji powiodo si
        if ((s.action == null) || s.action.performAction(req, context)) {
            if (currentState < states.length - 1) {
                s = states[++currentState];
            } else {
                currentState = 0;
                s = states[currentState];
            }
        }
        
        // zwraca adres uri dla biecego stanu
        return s.viewUri;
    }
    
    /**
     * Parsuje plik XML zawierajcy opis stanw
     */
    private State[] parseXML(InputStream is) throws Exception {
        DocumentBuilderFactory factory = DocumentBuilderFactory.newInstance();
        DocumentBuilder builder = factory.newDocumentBuilder();
        
        Document doc = builder.parse(is);
       
        // znajduje element opisujcy ciek zada
        NodeList workflows = doc.getElementsByTagName(WORKFLOW_TAG);
        Element workflow = (Element)workflows.item(0);
        
        // znajduje wszystkie stany
        NodeList states = doc.getElementsByTagName(STATE_TAG);
        State[] stateList = new State[states.getLength()];
        
        // wczytuje informacje o stanach
        for(int i = 0; i < states.getLength(); i++) {
            stateList[i] = new State();
            
            Element curState = (Element)states.item(i);
            stateList[i].name = curState.getAttribute(NAME_ATTR);
            stateList[i].viewUri = curState.getAttribute(VIEW_ATTR);
            
            // zamienia nazwy akcji na instancje klas
            String action = curState.getAttribute(ACTION_ATTR);
            if (action != null && action.length() > 0) {
                Class c = Class.forName(ACTION_PREFIX + action);
                stateList[i].action = (Action)c.newInstance();
            }
        }
        
        return stateList;
    }
}
