// Example 10-3: TransactionWrapper.java

package com.oreilly.patterns.chapter10;

import java.lang.reflect.InvocationHandler;
import java.lang.reflect.InvocationTargetException;
import java.lang.reflect.Method;
import java.lang.reflect.Proxy;

import java.sql.Connection;

final class TransactionWrapper {

    /**
     * Dekoruje obiekt penomocnika biznesowego za pomoc 
     * obiektu obudowujcego. Obiekt zwracany przez t metod bdzie
     * implementowa wszystkie interfejsy, ktre implementuje dekorowany
     * obiekt.
     *
     * @param obudowywany penomocnik biznesowy
     * @return obudowany penomocnik biznesowy
     */
    static Object decorate(Object delegate) {
        return Proxy.newProxyInstance(delegate.getClass().getClassLoader(),
            delegate.getClass().getInterfaces(),
            new XAWrapperHandler(delegate));
    }

    static final class XAWrapperHandler implements InvocationHandler {
        private final Object delegate;

        XAWrapperHandler(Object delegate) {
            // buforuje obudowanego penomocnika, 
            // aby przekaza mu wywoania metod
            this.delegate = delegate;
        }

        /** Wywouje metod wewntrz transakcji. Pobiera poczenie,
         * wycza automatyczne zatwierdzanie polece (tym samym rozpoczynajc transakcj), 
         * wykonuje oryginaln metod, zatwierdza transakcj, i zwraca wynik. Jeli
         * zostan wyrzucone jakie wyjtki (SQLException lub inne), to transakcja 
         * zostanie odwoana. 
         * 
         * Zauwamy, e blok finally nie przywraca trybu automatycznego zatwierdzania 
         * polece. Zastosowane tutaj rozwizanie daje nam lepsze pojcie o rzeczywistej
         * przyczynie bdu. W JDK 1.4, moglibymy zmodyfikowa blok catch tak, by
         * docza oryginalny obiekt Throwable jako przyczyn wyjtkw wyrzucanych
         * przez metody rollback() i setAutoCommit(). */
        public Object invoke(Object proxy, Method method, Object[] args)
            throws Throwable {
            Object result = null;
            Connection con = ConnectionManager.getConnection();
            try {
              con.setAutoCommit(false);
              result = method.invoke(delegate, args);
              con.commit();
              con.setAutoCommit(true);
            } catch (Throwable t) {
              // Wyjtek odwoania transakcji zostanie wyrzucony przez wywoywan metod
              con.rollback(); 
              con.setAutoCommit(true);
              throw t;
            } 
            
            return result;
        }
    }
}
