//
//  ICFGameCenterManager.m
//  WhackACac
//
//  Created by Kyle Richter on 7/3/12.
//  Copyright (c) 2012 Dragon Forged Software. All rights reserved.
//

#import "ICFGameCenterManager.h"

@implementation ICFGameCenterManager

@synthesize delegate;

static ICFGameCenterManager *sharedManager = nil;

#pragma mark -
#pragma mark Overhead

+ (ICFGameCenterManager *)sharedManager
{
    @synchronized(self)
    {
        if (sharedManager == nil)
        {
            sharedManager = [[self alloc] init];
        }
    }
    
    return sharedManager;
}

- (void)callDelegateOnMainThread:(SEL)selector withArg:(id)arg error:(NSError*)error
{
	dispatch_async(dispatch_get_main_queue(), ^(void)
    {
        [self callDelegate: selector withArg: arg error: error];
    });
}

- (void)callDelegate: (SEL)selector withArg: (id)arg error:(NSError*)error
{
	assert([NSThread isMainThread]);
    
    if(delegate == nil)
    {
        NSLog(@"Game Center Manager Delegate has not been set");
        return;
    }
    
	if([delegate respondsToSelector: selector])
	{
		if(arg != NULL)
        {
			[delegate performSelector: selector withObject: arg withObject: error];
        }
		
		else
        {
			[delegate performSelector: selector withObject: error];
        }
	}
	
	else
    {
		NSLog(@"Unable to find delegate method '%s' in class %@", sel_getName(selector), [delegate class]);
    }
}

-(void)dealloc
{
    if(achievementDictionary != nil)
    {
        [achievementDictionary release];
        achievementDictionary = nil;
    }
    
    [delegate release]; delegate = nil;
    [super dealloc];
}

#pragma mark -
#pragma mark Authentication 

- (void) authenticateLocalUser
{
	if([GKLocalPlayer localPlayer].authenticated == NO)
	{
		[[GKLocalPlayer localPlayer] authenticateWithCompletionHandler:^(NSError *error)
		 {
             if(error != nil)
             {
                 if([error code] == GKErrorNotSupported)
                 {
                     UIAlertView *alert = [[UIAlertView alloc] initWithTitle:@"Error" message:@"This device does not support Game Center" delegate:nil cancelButtonTitle:@"Dismiss" otherButtonTitles:nil];
                     [alert show];
                     [alert release];
                 }
                 
                 else if([error code] == GKErrorCancelled)
                 {
                     UIAlertView *alert = [[UIAlertView alloc] initWithTitle:@"Error" message:@"This device has failed login too many times from the app, you will need to login from the Game Center.app" delegate:nil cancelButtonTitle:@"Dismiss" otherButtonTitles:nil];
                     [alert show];
                     [alert release];
                 }
                 
                 else
                 {
                     UIAlertView *alert = [[UIAlertView alloc] initWithTitle:@"Error" message:[error localizedDescription] delegate:nil cancelButtonTitle:@"Dismiss" otherButtonTitles:nil];
                     [alert show];
                     [alert release];
                 }
                 
                 
                 return;
             }
             
			 [self callDelegateOnMainThread: @selector(gameCenterLoggedIn:) withArg: NULL error: error];
             [self submitAllSavedScores];
            
             //Remove comment to reset all achievements on authenication. 
             //[self resetAchievements];
             
             [self populateAchievementCache];
		 }];
	}
}

-(void)authenticateLocalUseriOSSix
{
    if([GKLocalPlayer localPlayer].authenticateHandler == nil)
	{
       [[GKLocalPlayer localPlayer] setAuthenticateHandler:^(UIViewController *viewController, NSError *error)
        {
            if(error != nil)
            {
                if([error code] == GKErrorNotSupported)
                {
                    UIAlertView *alert = [[UIAlertView alloc] initWithTitle:@"Error" message:@"This device does not support Game Center" delegate:nil cancelButtonTitle:@"Dismiss" otherButtonTitles:nil];
                    [alert show];
                    [alert release];
                }
                
                else if([error code] == GKErrorCancelled)
                {
                    UIAlertView *alert = [[UIAlertView alloc] initWithTitle:@"Error" message:@"This device has failed login too many times from the app, you will need to login from the Game Center.app" delegate:nil cancelButtonTitle:@"Dismiss" otherButtonTitles:nil];
                    [alert show];
                    [alert release];
                }
                
                else
                {
                    UIAlertView *alert = [[UIAlertView alloc] initWithTitle:@"Error" message:[error localizedDescription] delegate:nil cancelButtonTitle:@"Dismiss" otherButtonTitles:nil];
                    [alert show];
                    [alert release];
                }
            }

            else
            {
                if(viewController != nil)
                {
                    [(UIViewController *)delegate presentModalViewController:viewController animated:YES];
                }
                
                [self submitAllSavedScores];
                [self populateAchievementCache];

            }
        }];
    }
    
    else
    {
        [[GKLocalPlayer localPlayer] authenticate];
    }
}

#pragma mark -
#pragma mark Scores

- (void)reportScore:(int64_t)score forCategory:(NSString*)category
{
	GKScore *scoreReporter = [[[GKScore alloc] initWithCategory:category] autorelease];
	scoreReporter.value = score;
    
	[scoreReporter reportScoreWithCompletionHandler: ^(NSError *error)
	 {
		 if (error != nil)
		 {
			 NSData* savedScoreData = [NSKeyedArchiver archivedDataWithRootObject:scoreReporter];
             [self storeScoreForLater: savedScoreData];
		 }
		 
		 [self callDelegateOnMainThread:@selector(gameCenterScoreReported:) withArg:NULL error:error];
	 }];
}

- (void)storeScoreForLater:(NSData *)scoreData;
{
	NSMutableArray *savedScoresArray = [[NSMutableArray alloc] initWithArray: [[NSUserDefaults standardUserDefaults] objectForKey:@"savedScores"]];
	[savedScoresArray addObject: scoreData];
	[[NSUserDefaults standardUserDefaults] setObject:savedScoresArray forKey:@"savedScores"];
	[savedScoresArray release];
}

-(void)submitAllSavedScores
{
	NSMutableArray *savedScoreArray = [[NSMutableArray alloc] initWithArray: [[NSUserDefaults standardUserDefaults] objectForKey:@"savedScores"]];
	[[NSUserDefaults standardUserDefaults] removeObjectForKey: @"savedScores"];
	
	for(NSData *scoreData in savedScoreArray)
	{
		GKScore *scoreReporter = [NSKeyedUnarchiver unarchiveObjectWithData: scoreData];
		
		[scoreReporter reportScoreWithCompletionHandler: ^(NSError *error)
		 {
			 if (error != nil)
			 {
				 NSData* savedScoreData = [NSKeyedArchiver archivedDataWithRootObject:scoreReporter];
				 [self storeScoreForLater: savedScoreData];
			 }
			 
			 else
             {
				 NSLog(@"Successfully submitted scores that were pending submission");
                 [self callDelegateOnMainThread:@selector(gameCenterScoreReported:) withArg:NULL error:error];
             }
		 }];
	}
}

#pragma mark -
#pragma mark Achievements


- (void)resetAchievements
{
	[achievementDictionary removeAllObjects];
	
	[GKAchievement resetAchievementsWithCompletionHandler: ^(NSError *error)
	 {
		 if(error == nil)
		 {
			 NSLog(@"All achievements have been successfully reset");
		 }
         
		 else
		 {
			 NSLog(@"Unable to reset achievements: %@", [error localizedDescription]);
		 }
	 }];
}

-(void)populateAchievementCache
{
    if(achievementDictionary != nil)
    {
        NSLog(@"Repopulating achievement cache: %@", achievementDictionary);
    }
    
    else
    {
        achievementDictionary = [[NSMutableDictionary alloc] init];
    }
    
    [GKAchievement loadAchievementsWithCompletionHandler:^(NSArray *achievements, NSError *error)
     {
        
         if(error != nil)
         {
             NSLog(@"An error occurred while populating the achievement cache: %@", [error localizedDescription]);
         }
         
         else
         {
             for(GKAchievement *achievement in achievements)
             {
                 [achievementDictionary setObject:achievement forKey:[achievement identifier]];
             }
         }
     }];
}



-(void)reportAchievement:(NSString *)identifier withPercentageComplete:(double)percentComplete
{
    if(achievementDictionary == nil)
    {
        NSLog(@"An achievement cache must be populated before submitting achivement progress");
        return;
    }
        
    GKAchievement *achievement = [achievementDictionary objectForKey: identifier];
    
    if(achievement == nil) //no progress has yet been made on this achievement 
    {
        achievement = [[[GKAchievement alloc] initWithIdentifier: identifier] autorelease];
        [achievement setPercentComplete: percentComplete];
        [achievementDictionary setObject: achievement forKey:identifier];
    }
    
    else //this achievement has some progress
    {
        if([achievement percentComplete] >= 100.0 || [achievement percentComplete] >= percentComplete)
        {
            NSLog(@"Attempting to update achievement %@ which is either already completed or is decreasing percentage complete (%f)", identifier, percentComplete);
            return;
        }
        
        [achievement setPercentComplete: percentComplete];
        [achievementDictionary setObject: achievement forKey:identifier];
    }
    
    achievement.showsCompletionBanner = YES;
    
    [achievement reportAchievementWithCompletionHandler:^(NSError *error)
    {
        if(error != nil)
        {
            NSLog(@"There was an error submitting achievement %@: %@", identifier, [error localizedDescription]);
        }
        
        [self callDelegateOnMainThread:@selector(gameCenterAchievementReported:) withArg:NULL error:error];
    }];
}

-(GKAchievement *)achievementForIdentifier:(NSString *)identifier
{
    GKAchievement *achievement = nil;
    
    achievement = [achievementDictionary objectForKey:identifier];
    
    if(achievement == nil)
    {
        achievement = [[[GKAchievement alloc] initWithIdentifier:identifier] autorelease];
        [achievementDictionary setObject: achievement forKey:identifier];
    }
    
    return achievement;
}

@end
