// dołącz obsługę WiFi w zależności od zastosowanego układu ESP
#if defined(ESP8266)
  #include <ESP8266WiFi.h>       // funkcje sieciowe dla układu ESP8266
  #include <ESP8266WebServer.h>  // serwer WWW z obsługą żądań HTTP GET i POST
#elif defined(ESP32)
  #include <WiFi.h>              // funkcje sieciowe dla układu ESP32
  #include <WebServer.h>         // serwer WWW z obsługą żądań HTTP GET i POST
#endif

#define FS_ENABLE true           // czy aktywować obsługę systemu plików?

#if FS_ENABLE
  #include <LittleFS.h>          // dołącz obsługę systemu plików
#endif
#include <ArduinoOTA.h>          // i aktualizacji OTA

#define OTA_hostname "esp-01"    // nazwa urządzenia w sieci lokalnej
#define OTA_passwd "esp"         // hasło dostępu do OTA

#if defined(ESP8266)
  #define OTA_port 8266 // domyślnie dla ESP8266
#elif defined(ESP32)
  #define OTA_port 3232 // domyślnie dla ESP32
#endif

// utwórz obiekt serwera WWW na porcie 80
#if defined(ESP8266)
  ESP8266WebServer Server(80);
#elif defined(ESP32)
  WebServer Server(80);
#endif

char wifi_ssid[]   = "nazwa sieci WiFi";
char wifi_passwd[] = "hasło";

#define serialSpeed 115200 // prędkość transmisji UART

void setup()
{
  Serial.begin(serialSpeed);

  // połącz z punktem dostępowym
  Serial.print(F("Łączę się z siecią WiFi "));
  WiFi.begin(wifi_ssid, wifi_passwd);

  while (WiFi.status() != WL_CONNECTED) { // czekaj na połączenie
    delay(500);
    Serial.print(F("."));
  }

  Serial.println(F(" gotowe!"));

  #if FS_ENABLE
    // aktywuj obsługę systemu plików LittleFS
    Serial.print(F("Inicjalizacja systemu plików LittleFS ..."));
    if (LittleFS.begin()){
      Serial.println(F(" gotowe!"));
    } else {
      Serial.println(F(" coś poszło nie tak :("));
    }
  #endif
  
  // uruchom serwer WWW
  Server.begin();
  Serial.println(F("Serwer WWW uruchomiony."));
  Serial.println(F("Otwórz w przeglądarce jeden z poniższych adresów URL:"));
  Serial.print(F("=> http://"));
  Serial.println(WiFi.localIP());
  Serial.print(F("=> http://"));
  Serial.print(OTA_hostname);
  Serial.println(F(".local"));
  
  Server.on("/", handleRoot); // obsługa URL strony głównej

  // konfiguracja OTA
  ArduinoOTA.setPort(OTA_port);         // port OTA
  ArduinoOTA.setHostname(OTA_hostname); // nazwa urządzenia w sieci lokalnej
  ArduinoOTA.setPassword(OTA_passwd);   // hasło dostępu do OTA

  // funkcje obsługi OTA
  ArduinoOTA.onStart([]() { // rozpoczęcie przesyłania pliku
    String type;
    if (ArduinoOTA.getCommand() == U_FLASH) // firmware
      type = "firmware";
    else {                                  // system plików
      type = "system plików";
      #if FS_ENABLE
        LittleFS.end();                     // odmontuj system plików
      #endif
    }
    Serial.print(F("Typ aktualizacji: "));
    Serial.println(type);
  });
  
  ArduinoOTA.onEnd([]() { // zakończenie przesyłania pliku
    Serial.println(F("Gotowe!"));
  });

  // postęp przesyłania pliku
  ArduinoOTA.onProgress([](unsigned int progress, unsigned int total) {
    Serial.printf("Postęp: %u%%\n", (progress / (total / 100)));
  });

  // obsługa wystapienia błędu podczas przesyłania pliku
  ArduinoOTA.onError([](ota_error_t error) {
    Serial.printf("Błąd[%u]: ", error);
  });

  // aktywacja OTA
  ArduinoOTA.begin();
}

void loop() 
{
  Server.handleClient(); // czekaj na połączenia klientów
  ArduinoOTA.handle();   // obsłuż żądanie OTA
}

// obsługa wywołania URL strony głównej
void handleRoot()
{
  Serial.println(F("=> Odebrano żądanie GET /"));
  Server.send(200, "text/html", F("Serwer WWW aktywny!"));
}
