// dołącz obsługę WiFi w zależności od zastosowanego układu ESP
#if defined(ESP8266)
  #include <ESP8266WiFi.h>       // funkcje sieciowe dla układu ESP8266
  #include <ESP8266WebServer.h>  // serwer WWW z obsługą żądań HTTP GET i POST
  #include <ESP8266mDNS.h>       // włącz obsługę multicast DNS 
  #include <ESP8266HTTPUpdateServer.h> // i aktualizacji przez OTA
#elif defined(ESP32)
  #include <WiFi.h>              // funkcje sieciowe dla układu ESP32
  #include <WebServer.h>         // serwer WWW z obsługą żądań HTTP GET i POST
  #include <ESPmDNS.h>           // włącz obsługę multicast DNS
  #include <Update.h>            // i aktualizacji przez OTA
#endif

#include <LittleFS.h>            // dołącz obsługę systemu plików LittleFS
#include "DHT.h"                 // i czujnika DHT

#define DHTPIN D2     // numer GPIO dla linii sygnałowej DAT
#define DHTTYPE DHT22 // typ czujnika DHT
/*
#define DHTTYPE DHT11
#define DHTTYPE DHT21 // DHT 21 (AM2301)
#define DHTTYPE DHT22 // DHT 22 (AM2302), AM2321
*/

DHT dht(DHTPIN, DHTTYPE); // obiekt obsługujący czujnik DHT

// utwórz obiekt serwera WWW na porcie 80
#if defined(ESP8266)
  ESP8266WebServer Server(80);
  ESP8266HTTPUpdateServer httpUpdater;
#elif defined(ESP32)
  WebServer Server(80);
#endif

// dostęp do sieci WiFi
char wifi_ssid[]   = "nazwa sieci WiFi";
char wifi_passwd[] = "hasło";

// autoryzacja dostępu do strony WWW 
// umożliwiającą aktualizację oprogramowania
bool require_auth = true;   // czy włączyć obsługę HTTP Authentication?
bool http_auth = false;     // czy nazwę użytkownika i hasło wpisano poprawnie?
String http_user = "esp";   // nazwa użytkownika
String http_passwd = "esp"; // hasło 

#define serialSpeed 115200 // prędkość transmisji UART
#define mDNS_host "esp-01" // nazwa urządzenia w sieci lokalnej - http://esp-01.local

void setup()
{
  Serial.begin(serialSpeed);

  Serial.print(F("Inicjalizacja czujnika DHT ... "));
  dht.begin();
  Serial.println(F("gotowe!"));

  // połącz z punktem dostępowym
  Serial.print(F("Łączę się z siecią WiFi "));
  WiFi.begin(wifi_ssid, wifi_passwd);

  while (WiFi.status() != WL_CONNECTED) { // czekaj na połączenie
    delay(500);
    Serial.print(F("."));
  }

  Serial.println(F(" gotowe!"));

  // aktywuj usługę multicast DNS
  if (MDNS.begin(mDNS_host)) { 
    Serial.println(F("Usługa mDNS aktywna."));
  }

  // aktywuj obsługę systemu plików LittleFS
  Serial.print(F("Inicjalizacja systemu plików LittleFS ..."));
  if (LittleFS.begin()){
    Serial.println(F(" gotowe!"));
  } else {
    Serial.println(F(" coś poszło nie tak :("));
  }

  // obsługa wywołania URL strony głównej
  Server.on("/", HTTP_GET, []() {
    Serial.println(F("=> Odebrano żądanie GET /"));
    Server.sendHeader("Connection", "close");
    // odczytaj plik i prześlij kod HTML do użytkownika 
    loadFromFS("/index.html");
  });

  // obsługa wywołania URL do odczytu danych z czujnika
  Server.on("/data", HTTP_GET, []() {
    Serial.println(F("=> Odebrano żądanie GET /data"));
    Server.sendHeader("Connection", "close");
    String data = getTemperatureHumidit(true); // odczyt danych z czujnika
    Server.send(200, "text/plain", data);
  });

  // obsługa wywołania URL strony do aktualizacji oprogramowania
  Server.on("/update", HTTP_GET, []() {
    Serial.println(F("=> Odebrano żądanie GET /update"));
    // sprawdź czy wymagana jest autoryzacja użytkownika
    if (require_auth) {
      if(
        http_user != emptyString && http_passwd != emptyString && 
        !Server.authenticate(http_user.c_str(), http_passwd.c_str())
      ) {
        Serial.println(F("=> Odebrano żądanie GET /update => wymagana autoryzacja"));
        return Server.requestAuthentication();
      }
    }
    Server.sendHeader("Connection", "close");
    // odczytaj plik i prześlij kod HTML do użytkownika    
    loadFromFS("/update.html");
  });  

  // obsługa danych wysłanych z formularza w trakcie aktualizacji oprogramowania
  Server.on("/update", HTTP_POST, [&]() {
    if(require_auth && !http_auth) { // czy użytkownik ma autoryzację?
      Serial.println(F("=> Odebrano żądanie POST /update => wymagana autoryzacja"));
      return Server.requestAuthentication();
    }
    Server.send(200, "text/plain", (Update.hasError()) ? "FAIL" : "OK");
    ESP.restart();
  }, [&]() {
    // obsługa przesłanych danych - skompilowanego pliku binarnego 
    HTTPUpload& upload = Server.upload();
    
    if (upload.status == UPLOAD_FILE_START) { // przesyłanie danych do pamięci
      Serial.printf("Odbieram plik: %s\n", upload.filename.c_str());

      #if defined(ESP8266)
        WiFiUDP::stopAll();
      #endif

      // czy użytkownik ma autoryzację?
      if (require_auth) {
        http_auth = (
          http_user == emptyString || http_passwd == emptyString || 
          Server.authenticate(http_user.c_str(), http_passwd.c_str())
        );
        if(!http_auth){
          Serial.println(F("Aktualizacja niemożliwa, brak autoryzacji."));
          return;        
        }
      }

      // oblicz maksymalną wielkość pliku, który można wgrać
      Serial.print(F("=> Aktualizacja "));
      if (upload.name == "firmware") {
        Serial.println(F("oprogramowania układowego (firmware)"));
        uint32_t maxSketchSpace = (ESP.getFreeSketchSpace() - 0x1000) & 0xFFFFF000;
        if (!Update.begin(maxSketchSpace, U_FLASH)) { // udostępnij całą pamięć
          Update.printError(Serial);
        }        
      } else {
        Serial.println(F("systemu plików LittleFS"));
        #if defined(ESP8266)
          size_t fsSize = ((size_t) &_FS_end - (size_t) &_FS_start);
          close_all_fs();
          if (!Update.begin(fsSize, U_FS)) { // udostępnij całą pamięć
            Update.printError(Serial);
          }            
        #elif defined(ESP32)
          size_t fsSize = UPDATE_SIZE_UNKNOWN;
          LittleFS.end(); // odmontuj system plików
          if (!Update.begin(fsSize, U_SPIFFS)) { // udostępnij całą pamięć
            Update.printError(Serial);
          }
        #endif          
      }
         
    } else if (upload.status == UPLOAD_FILE_WRITE) {
      // wgrywanie oprogramowania do pamięci układu - flashowanie
      if (Update.write(upload.buf, upload.currentSize) != upload.currentSize) {
        Update.printError(Serial);
      }
    } else if (upload.status == UPLOAD_FILE_END) {
      if (Update.end(true)) { // czy wszystko wgrane?
        Server.send(200, "text/html", F("Gotowe!"));        
        Serial.printf("Wielkość: %u bajtów\nRestart modułu ...\n", upload.totalSize);
      } else {
        Update.printError(Serial);
      }
    }
  });

  #if defined(ESP8266)
    httpUpdater.setup(&Server);
  #endif
  Server.begin();
  MDNS.addService("http", "tcp", 80);
  
  Serial.println(F("Serwer WWW uruchomiony."));
  Serial.println(F("Otwórz w przeglądarce jeden z poniższych adresów URL:"));
  Serial.print(F("=> http://"));
  Serial.println(WiFi.localIP());
  Serial.print(F("=> http://"));
  Serial.print(mDNS_host);
  Serial.println(F(".local"));  
}

void loop() 
{
  Server.handleClient(); // czekaj na połączenia klientów
  #if defined(ESP8266)
    MDNS.update();       // aktualizuj nazwę urządzenia w sieci lokalnej
  #endif
}

// odczytanie pliku z pamięci i wysłanie do użytkownika
bool loadFromFS(String path) 
{
  Serial.print(F("=> Odczytuję z pamięci plik "));
  Serial.print(path);
  if (LittleFS.exists(path))
  {
    File dataFile = LittleFS.open(path, "r");
    
    if (!dataFile) {
      handleNotFound();
      return false;
    }

    String contentType = getContentType(path);
    
    if (Server.streamFile(dataFile, contentType) != dataFile.size()) {
      Serial.println(F(" ... coś poszło nie tak :("));
    } else {
      Serial.println(F(" ... gotowe!"));
    }
    dataFile.close();
  } else {
    handleNotFound();
    return false;
  }
  return true;
}

// sprawdź typ pliku wysyłanego do użytkownika
String getContentType(String filename)
{
  String dataType = "text/plain";
  
  if (Server.hasArg("download")) dataType = "application/octet-stream";
  else if (filename.endsWith(".html") || filename.endsWith(".html.gz")) dataType = "text/html";
  else if (filename.endsWith(".css") || filename.endsWith(".css.gz")) dataType = "text/css";
  else if (filename.endsWith(".js") || filename.endsWith(".js.gz"))  dataType = "application/javascript";
  else if (filename.endsWith(".png")) dataType = "image/png";
  else if (filename.endsWith(".gif")) dataType = "image/gif";
  else if (filename.endsWith(".jpg")) dataType = "image/jpeg";
  else if (filename.endsWith(".ico")) dataType = "image/x-icon";
  else if (filename.endsWith(".xml") || filename.endsWith(".xml.gz")) dataType = "text/xml";
  else if (filename.endsWith(".pdf")) dataType = "application/x-pdf";
  else if (filename.endsWith(".zip")) dataType = "application/x-zip";
  else if (filename.endsWith(".gz"))  dataType = "application/x-gzip";

  return dataType;
}

// obsługa błędu HTTP 404
void handleNotFound()
{
  String message = "404 Not Found\n\n";
  message += "The request ";
  message += Server.uri();
  message += " was not found on this server.\n";
  message += "\nMethod: ";
  message += (Server.method() == HTTP_GET) ? "GET" : "POST";
  message += "\nArguments: ";
  message += Server.args();
  message += "\n";
  for (int i = 0; i < Server.args(); i++)
  {
    message += " " + Server.argName(i) + ": " + Server.arg(i) + "\n";
  }
  Server.send(404, "text/plain", message);
}

// odczyt danych z czujnika
String getTemperatureHumidit(bool debug)
{
  float temp = dht.readTemperature();
  float hmd = dht.readHumidity();

  if (!(isnan(temp) || isnan(hmd))){
    if (debug)
    {
      Serial.print(F("Odczyt danych z czujnika DHT => T: "));
      Serial.print(temp);
      Serial.print(F("°C, H: "));
      Serial.print(hmd);
      Serial.println(F("%"));
    }
  } else {
    temp = 0;
    hmd = 0;
    Serial.println(F("Błędny odczyt z czujnika DHT."));
  }
    
  String data = String(temp) + String(",") + String(hmd);
  return data; // zwróć wartość temperatury i wilgotności
}
