#include <ESP8266WiFi.h>       // funkcje sieciowe dla układu ESP8266
#include <ESP8266WebServer.h>  // serwer WWW z obsługą żądań HTTP GET i POST
#include <ESP8266mDNS.h>       // włącz obsługę multicast DNS
#include <OneWire.h>           // obsługa interfejsu 1-wire
#include <DallasTemperature.h> // i czujnika DS18B20

#define serialSpeed 115200     // prędkość transmisji UART

char wifi_ssid[]   = "nazwa sieci WiFi";
char wifi_passwd[] = "hasło";

#define ONE_WIRE_BUS 2          // czujnik DS18B20 podpięty do GPIO2
OneWire oneWire(ONE_WIRE_BUS);  // utwórz obiekt 1-wire
DallasTemperature ds(&oneWire); // do komunikacji z DS18B20

ESP8266WebServer Server(80);    // obiekt serwera WWW na porcie 80

char* mDNS_host = "esp8266";    // nazwa hosta w sieci lokalnej
float temp = 0;

// kod HTML strony startowej zapisany w pamięci programu (nie w pamięci RAM)
const char page_index[] PROGMEM = R"=====(
<!doctype html>
<html>
<head>
<meta charset="utf-8">
<title>Mój Smart Home</title>
<style>
body { padding: 2rem; }
h1 { font-size: 2rem; }
h2 { font-size: 1rem; }
.item { font-size: 2.5rem; }
.item .unit { font-size: 1.0rem; margin-left: 5px; }
</style>
</head>
<body>
<h1>Mój Smart Home</h1>
<h2>Pomiar temperatury</h2>
<p class="item">Temperatura <span id="temp">0.0</span><sup class="unit">°C</sup></p>
<hr>
<p><button onclick="getData()">Odczyt temperatury</button></p>
</body>
<script>
function getData(){
 var xhttp = new XMLHttpRequest();
 xhttp.onreadystatechange = function(){
  if (this.readyState == 4 && this.status == 200){
   data = this.responseText;
   document.getElementById('temp').innerHTML = data;
  }
 };
 xhttp.open('GET', '/data', true);
 xhttp.send();
};
getData();
</script>
</html>
)=====";

void setup() 
{
  Serial.begin(serialSpeed);

  Serial.print(F("Inicjalizacja czujnika DS18B20 ... "));
  ds.begin();
  Serial.println(F("gotowe!"));

  // połącz z punktem dostępowym
  Serial.print(F("Łączę się z siecią WiFi "));
  WiFi.begin(wifi_ssid, wifi_passwd);

  while (WiFi.status() != WL_CONNECTED) { // czekaj na połączenie
    delay(500);
    Serial.print(F("."));
  }

  Serial.println(F(" gotowe!"));

  // aktywuj usługę multicast DNS
  if (MDNS.begin(mDNS_host)) { 
    Serial.println(F("Usługa mDNS aktywna."));
  }

  // uruchom serwer WWW
  Server.begin();
  Serial.println(F("Serwer WWW uruchomiony."));
  Serial.println(F("Otwórz w przeglądarce jeden z poniższych adresów URL:"));
  Serial.print(F("=> http://"));
  Serial.println(WiFi.localIP());
  Serial.print(F("=> http://"));
  Serial.print(mDNS_host);
  Serial.println(F(".local"));

  // obsługa adresów URL
  Server.on("/", handleRoot);        // URL strony głównej
  Server.on("/data", handleData);    // URL do odczytu temperatury
  Server.onNotFound(handleNotFound); // obsługa błędu HTTP 404
}

void loop() 
{
  Server.handleClient(); // czekaj na połączenia klientów
  MDNS.update();
}

float getTemperature(bool debug)
{
  ds.requestTemperatures(); // wyślij żądanie odczytu temperatury
  float tempC = ds.getTempCByIndex(0); // odczytaj wartość temperatury

  if (tempC != DEVICE_DISCONNECTED_C) { // jeśli nie pojawił się błąd
    if (debug) {
      Serial.print(F("Odczyt temperatury z urządzenia nr 1 (indeks 0): "));
      Serial.print(tempC);
      Serial.println(F("°C"));
    }
  } else { // jeśli coś poszło nie tak
    if (debug) {
      Serial.println(F("Błąd odczytu danych z urządzenia nr 1"));
    }
  }

  return tempC; // zwróć wartość temperatury
}

// obsługa wywołania URL strony głównej
void handleRoot()
{
  Serial.println(F("=> Odebrano żądanie GET /"));
  Server.send(200, "text/html", page_index);
}

// obsługa wywołania URL do odczytu temperatury
void handleData()
{
  Serial.println(F("=> Odebrano żądanie GET /data"));
  temp = getTemperature(true);
  Server.send(200, "text/plain", String(temp));
}

// obsługa błędu HTTP 404
void handleNotFound() 
{
  String message = "404 Not Found\n\n";
  message += "The request ";
  message += Server.uri();
  message += " was not found on this server.\n";
  message += "\nMethod: ";
  message += (Server.method() == HTTP_GET) ? "GET" : "POST";
  message += "\nArguments: ";
  message += Server.args();
  message += "\n";
  for (int i = 0; i < Server.args(); i++)
  {
    message += " " + Server.argName(i) + ": " + Server.arg(i) + "\n";
  }
  Server.send(404, "text/plain", message);
}
