/**
* Opis:
*   Klasa SVGChart sluzy do wizualizacji danych liczbowych w postaci wykresu utworzonego
*   w technologii SVG
*
* Wymagania:
*   Klasa SVGChart wymaga do funkcjonowania skryptu z klasą SVGDraw
*
* Uzycie:
*   Tworzenie nowego wykresu polega na wywolaniu konstruktora klasy SVGChart jak ponizej:
*     new SVGChart(@ustawienia Object)
*   i dodaniu do tak utworzonej instancji klasy serii danych liczbowych (z i/lub dodatkowa
*   etykieta tekstowa) za pomoca metody addData:
*     addData(@dane do wyswietlenia Array [, @ustawienia Object])
*   Format danych i ustawienia przekazane do metody addData sa zalezne od typu wykresu
*
* Argument konstruktora:
*   Argumentem konstruktora jest obiekt w formacie JSON. Jedynym wymaganym parametrem
*   jest kontener (najczesciej element DIV w dokumencie HTML) w ktorym wykres ma zostac
*   utworzony.
*
*   Parametry:
*        ************************************************************************************
*        * Parametr	      *  Wartosci		      *		Opis		    *
*        ************************************************************************************
*        * cntID		identyfikator String |		Element w dokumencie HTML   *
*        * 			referencja do elementu Object   w ktorym wykres jest ryso-  *
*        * 							wany			    *
*        ************************************************************************************
*        * chartType		"bar" | "pie" | "line"		Typ tworzonego wykresu      *
*        ************************************************************************************
*        * fontFamily		rodzina czcionki String		Rodzina czcionki stosowana  *
*        * 							przy tworzeniu etykiet i    *
*        * 							wyswietlaniu wartosci       *
*        ************************************************************************************
*        * fontSize		wartosc Int |			Wielkosc czcionki stosowana *
*        * 			wartosc String			przy wyswietlaniu elementow *
*        * 							tekstowych (dla	            *
*        * 							ktorych nie zostala okre-   *
*        * 							slona wielkosc czcionki)    *
*        ************************************************************************************
*        * fontColor		"random" |			Barwa elementow tekstowych  *
*        * 			kolor String |						    *
*        * 			gradient Object (obiekt					    *
*        * 			zwrocony przez metode klasy				    *
*        * 			SVGDraw tworzaca gradient)				    *
*        ************************************************************************************
*        * fontWeight		"normal" | "bold" | "bolder" |	Waga czcionki elementow	    *
*        * 			"lighter" | wartosc Int		tekstowych dla ktorych waga *
*        * 							czcionki nie zostala okre-  *
*        * 							slona			    *
*        ************************************************************************************
*        * backgroundColor	"random" |			Barwa tla wykresu	    *
*        * 			kolor String |						    *
*        * 			gradient Object (obiekt					    *
*        * 			zwrocony przez metode klasy				    *
*        * 			SVGDraw tworzaca gradient)				    *
*        ************************************************************************************
*        * borderWidth		wartosc Int			Szerokosc konturu tla	    *
*        ************************************************************************************
*        * borderColor		"random" |			Barwa konturu tla	    *
*        * 			kolor String |						    *
*        * 			gradient Object (obiekt					    *
*        * 			zwrocony przez metode klasy				    *
*        * 			SVGDraw tworzaca gradient)				    *
*        ************************************************************************************
*        * borderRadius		wartosc Int			Promien zaokraglenia naro-  *
*        * 							znikow tla		    *
*        ************************************************************************************
*        * marginX		wartosc Int			Poziome odsuniecie wykresu  *
*        * 							od krawedzi tla		    *
*        ************************************************************************************
*        * marginY		wartosc Int			Pionowe odsuniecie wykresu  *
*        * 							od krawedzi tla		    *
*        ************************************************************************************
*        * chartLabel		etykieta String			Etykieta wykresu	    *
*        ************************************************************************************
*        * chartLabelFontWeight "normal" | "bold" | "bolder" |	Waga czcionki etykiety wy-  *
*        * 			"lighter" | wartosc Int		kresu			    *
*        * 										    *
*        ************************************************************************************
*        * chartLabelFontSize   wartosc Int |			Wielkosc czcionki etykiety  *
*        * 			wartosc String			wykresu			    *
*        ************************************************************************************
*        * chartLabelFontColor  "random" |			Barwa etykiety wykresu	    *
*        * 			kolor String |						    *
*        * 			gradient Object (obiekt					    *
*        * 			zwrocony przez metode klasy				    *
*        * 			SVGDraw tworzaca gradient)				    *
*        ************************************************************************************
*        * gridColor		"random" |			Barwa linii siatki	    *
*        * 			kolor String |						    *
*        * 			gradient Object (obiekt					    *
*        * 			zwrocony przez metode klasy				    *
*        * 			SVGDraw tworzaca gradient)				    *
*        ************************************************************************************
*        * gridWidth		wartosc Int			Szerokosc linii siatki	    *
*        ************************************************************************************
*        * gridOpacity		wartosc Int			Krycie linii siatki	    *
*        ************************************************************************************
*	 * gridFontSize		wartosc Int |			Wielkosc czcionki elementow *
*	 * 			wartosc String			tekstowych siatki	    *
*        ************************************************************************************
*        * gridXmargin		wartosc Int			Poziome odsuniecie siatki   *
*        * 							od obszaru przeznaczonego   *
*        * 							na wykres		    *
*        ************************************************************************************
*        * gridYmargin		wartosc Int			Pionowe odsuniecie siatki   *
*        * 							od obszaru przeznaczonego   *
*        * 							na wykres		    *
*        ************************************************************************************
*        * axesXLabel		etykieta String			Etykieta osi x		    *
*        ************************************************************************************
*        * axesYLabel		etykieta String			Etykieta osi y		    *
*        ************************************************************************************
*        * axesColor		"random" |			Barwa osi	    	    *
*        * 			kolor String |						    *
*        * 			gradient Object (obiekt					    *
*        * 			zwrocony przez metode klasy				    *
*        * 			SVGDraw tworzaca gradient)				    *
*        ************************************************************************************
*        * axesWidth		wartosc Int			Szerokosc linii osi	    *
*        ************************************************************************************
*        * axesLabelFontSize	wartosc Int			Wielkosc czcionki etykiet   *
*        * 							osi 			    *
*        ************************************************************************************
*
*
* Format danych i ustawienia wykresu slupkowego:
*   Format danych:
*     Array(
*       Array(wartosc-1 Int [, etykieta-1 String])
*	Array(wartosc-2 Int [, etykieta-2 String])
*	...
*	Array(wartosc-n Int [, etykieta-n String])
*	)
*	    
*   Ustawienia (format JSON):
*     {parametr-1: wartosc-1, parametr-2: wartosc-2 ..., parametr-n: wartosc-n}
*
*        ************************************************************************************
*        * Parametr	      *  Wartosci		     *		Opis		    *
*        ************************************************************************************
*        * barColor		"random" |			Barwa slupkow		    *
*        * 			kolor String |				 	    	    *
*        * 			gradient Object (obiekt					    *
*        * 			zwrocony przez metode klasy				    *
*        * 			SVGDraw tworzaca gradient)				    *
*        ************************************************************************************
*        * barRadius		wartosc Int		    	Promien zaokraglenia	    *
*        * 							naroznikow slupka	    *
*        ************************************************************************************
*        * barBorderWidth	wartosc Int			Szerokosc konturu slupka    *
*        ************************************************************************************
*        * barBorderColor	"random" |			Barwa konturu slupka	    *
*        * 			kolor String |						    *
*        * 			gradient Object (obiekt					    *
*        * 			zwrocony przez metode klasy				    *
*        * 			SVGDraw tworzaca gradient)				    *
*        ************************************************************************************
*        * barWidth		wartosc Int			Szerokosc slupka	    *
*        ************************************************************************************
* 	
* 
* Format danych i ustawienia wykresu liniowego:
*   Format danych:
*     Array(wartosc-1 Int, wartosc-2 Int ..., wartosc-n Int)
*	
*   Ustawienia (format JSON):
*     {parametr-1: wartosc-1, parametr-2: wartosc-2 ..., parametr-n: wartosc-n}
*
*        ************************************************************************************
*        * Parametr	      *  Wartosci		     *		Opis		    *
*        ************************************************************************************
*        * lineColor		"random" |			Barwa linii	 	    *
*        * 			kolor String |						    *
*        * 			gradient Object (obiekt					    *
*        * 			zwrocony przez metode klasy				    *
*        * 			SVGDraw tworzaca gradient)				    *
*        ************************************************************************************
*        * lineWidth		wartosc Int		    	Szerokosc linii	 	    *
*        ************************************************************************************
*        * showPoints		true | false			Pokazanie/ukrycie punktow   *
*        * 							wykresu			    *
*        ************************************************************************************
*        * pointsRadius		wartosc Int			Promien punktow wykresu	    *
*        ************************************************************************************
*        * fillChart		true | false			Wypelnienie/brak wypel-	    *
*        * 							nienia kolorem obszaru	    *
*        * 							pod wykresem		    *
*        ************************************************************************************
*        * fillOpacityChart	wartosc Int			Krycie barwy obszaru pod    *
*        * 							wykresem		    *
*        ************************************************************************************
*        * showLine		true | false			Pokazanie/ukrycie linii     *
*        * 							laczacej punkty wykresu     *
*        ************************************************************************************
*        * spacingXlength 	wartosc Int			Liczba podzialek wzdluz     *
*        * 							osi x			    *
*        ************************************************************************************
*        * spacingYlength 	wartosc Int			Liczba podzialek wzdluz     *
*        * 							osi y       		    *
*        ************************************************************************************
*        * spacingX		wartosc Int			Wielkosc podzialki wzdluz   *
*        * 							osi x			    *
*        ************************************************************************************
*        * spacingY		wartosc Int			Wielkosc podzialki wzdluz   *
*        * 							osi y			    *
*        ************************************************************************************
*        * showGrid 		true | false			Pokazanie/ukrycie siatki    *
*        ************************************************************************************
*        * showAxes		true | false			Pokazanie/ukrycie osi	    *
*        ************************************************************************************
*        * showAxesLabels	true | false			Pokazanie/ukrycie etykiet   *
*        * 							osi			    *
*        ************************************************************************************
*        * gridValuesPrecision  wartosc Int			Precyzja wartosci wyswie-   *
*        * 							tlanych wzdluz siatki	    *
*        ************************************************************************************
*
* 
* Format danych i ustawienia wykresu kolowego:
*   Format danych:
*     Array(
*       Array(wartosc-1 Int [, etykieta-1 String])
*	Array(wartosc-2 Int [, etykieta-2 String])
*	...
*	Array(wartosc-n Int [, etykieta-n String])
*	)
*	    
*   Ustawienia (format JSON):
*     {parametr-1: wartosc-1, parametr-2: wartosc-2 ..., parametr-n: wartosc-n}
*            
*        ************************************************************************************
*        * Parametr	      *  Wartosci		     *		Opis		    *
*        ************************************************************************************
*        * pieColor		"random" |			Barwa poszczegolnych	    *
*        * 			kolor String |			wycinkow wykresu	    *
*        * 			gradient Object (obiekt					    *
*        * 			zwrocony przez metode klasy				    *
*        * 			SVGDraw tworzaca gradient)				    *
*        ************************************************************************************
*        * pieBorderWidth	wartosc Int		    	Szerokosc linii	     	    *
*        * 						    	otaczajacej poszczegolne    *
*        * 						    	wycinki wykresu             *
*        ************************************************************************************
*        * pieBorderColor	"random" |			Barwa linii otaczajacej	    *
*        * 			kolor String |			poszczegolne fragmenty      *
*        * 			gradient Object (obiekt		wykresu			    *
*        * 			zwrocony przez metode klasy				    *
*        * 			SVGDraw tworzaca gradient)				    *
*        ************************************************************************************
*        * chartScale		wartosc Int		    	Stosunek srednicy	    *
*        * 						    	wykresu do wysokosci	    *
*        * 						    	obszaru przeznaczonego	    *
*        * 						    	na wykres		    *
*        ************************************************************************************
*
* @author Krzysztof Ciesla
* @license GNU
*
*/

var SVGChart = (function() {
    var methods = {
            /* Draw background and label */
            "initDraw": function() {
                var p = this.propObj;
                /* Draw background */
                this.bg = this.cnt.drawRectangle({
                    "fill"        : getColor.call(this, p.backgroundColor),
                    "width"       : this.cntWidth  - p.borderWidth,
                    "height"      : this.cntHeight - p.borderWidth,
                    "left"        : p.borderWidth/2,
                    "top"         : p.borderWidth/2,
                    "strokeWidth" : p.borderWidth,
                    "stroke"      : p.borderColor,
                    "radiusX"     : p.borderRadius
                    });
                /* Draw chart label */
                var setText = function(txt,l,t) {
                    return this.cnt.drawText(txt, {
                        "top"    : t,
                        "left"   : l,
                        "size"   : p.chartLabelFontSize,
                        "weight" : p.chartLabelFontWeight,
			"fill"	 : p.chartLabelFontColor,
                        "family" : p.fontFamily
                        })
                    };
                /* Align chart label */
                this.cnt.align(
                    setText.call(this, p.chartLabel),
                    this.bg,"center-top",0,5);

                },// END initDraw
            /* Draw axis, grid and labels */
            "drawGridAndAxes" : function() {
                var p = this.propObj, cp = this.chartPropObj;
                /* Draw grid */
		if(cp.showGrid) {
		    var d = "",
			h = cp.gridHeight,
			w = cp.gridWidth,
			v, txtElem, x, y,
			txtPropObj = {
                        "top"    : 0,
                        "left"   : 0,
                        "size"   : p.gridFontSize,
                        "weight" : p.fontWeight,
                        "family" : p.fontFamily
                        },
			round = function(v, precision) {
			    var n = Math.pow(10, precision);
			    return Math.round(v * n) / n;
			    };			

		    for(var i=0; i<=cp.gridsX; i++) {
			v = cp.gridXstartValue + (i * cp.spacingX);
			x = Math.round(cp.getXpos(v));		
			d += "M " + x + " " + (p.marginY + p.gridYmargin) + "v " + h + " ";	
                        txtElem = this.cnt.drawText(round(v, cp.gridValuesPrecision).toString(), txtPropObj);
                        txtElem.setAttribute("x", x - txtElem.getBBox().width/2);
                        txtElem.setAttribute("y", this.cntHeight - p.marginY + txtElem.getBBox().height + 2);
			};

		    for(var i=0; i<=cp.gridsY; i++) {
			v = cp.gridYstartValue + (i * cp.spacingY);
			y = Math.round(cp.getYpos(v));
			d += "M " + (p.marginX + p.gridXmargin) + " " + y + "h " + w;
                        txtElem = this.cnt.drawText(round(v, cp.gridValuesPrecision).toString(), txtPropObj);
                        txtElem.setAttribute("x", p.marginX - txtElem.getBBox().width - 10);
                        txtElem.setAttribute("y", y + txtElem.getBBox().height/2 - 3);
			};
			
		    var path = this.cnt.drawPath({
			"data"          : d+" Z",
			"stroke"        : p.gridColor,
			"strokeWidth"   : p.gridWidth,
                        "strokeOpacity" : p.gridOpacity,
                        "fill"          : "none"
			});
		    this.cnt.setTransform(path, {"transformName":"translate","tx":p.gridWidth/2,"ty":p.gridWidth/2});
		    };
                /* Draw axes */
                if(cp.showAxes) {
                    var d = "M "+p.marginX+" "+(p.marginY-4)+" v "+(this.height+4)+" h "+(this.width+4)+" M "+p.marginX+" "+(p.marginY-4);
                    this.cnt.drawPath({
                        "data"        : d,
                        "strokeWidth" : p.axesWidth,
                        "stroke"      : p.axesColor,
                        "fill"        : "none",
                        "strokeLinecap"  :"round",
                        "strokeLinejoin" :"bevel"
                        });
                    };
                /* Draw axes labels */
                if(cp.showAxesLabels) {
		    
		    txtPropObj.size = p.axesLabelFontSize;
		    txtElem = this.cnt.drawText(p.axesXLabel, txtPropObj);
                    this.cnt.align(
			txtElem,
			this.bg, "center-bottom");
                    this.cnt.align(
                        this.cnt.drawText(p.axesYLabel, txtPropObj),
                        this.bg, "left-middle", 5, 0);
                    };
                },// End drawGridAndAxes
            "drawChart": {
                "bar": function() {
                    var v = this.values, p = this.propObj, len = v.length, cp = this.chartPropObj;
                    var setText = function(txt,l,t) {
                    return this.cnt.drawText(txt, {
                        "top"    : t,
                        "left"   : l,
                        "size"   : p.fontSize,
                        "weight" : p.fontWeight,
                        "family" : p.fontFamily
                        })
                    }, bar, txt;

                    for(var i=0; i<len; i++) {
                        bar = this.cnt.drawRectangle({
                            "left"        :v[i].x,
                            "top"         :v[i].y,
                            "width"       :cp.barWidth,
                            "height"      :v[i].height,
                            "fill"        :v[i].color,
                            "radiusX"     :cp.barRadius,
                            "stroke"      :cp.barBorderColor,
                            "strokeWidth" :cp.barBorderWidth
                            });

                        txt = setText.call(this, v[i].label, 0, 0);
                        this.cnt.align(txt,bar,"center-bottom",0, 4+txt.getBBox().height);
                        //this.cnt.rotate(txt,35,"left-top");
                        txt = setText.call(this, v[i].value, 0, 0);
                        this.cnt.align(txt,bar,"center-top", 0, -4-txt.getBBox().height);
                        };
                    },
		"line": function() {
		    var p = this.propObj, cp = this.chartPropObj,
                        wspY = this.height/(cp.yMax - cp.yMin),
                        d = "", pos = [], x, y, v = this.values, len = v.length,
                        layer = this.cnt.addLayer(), color = getColor.call(this, cp.lineColor), point;

                    for(var i=0; i<len; i++) {
			
                        x = cp.getXpos(v[i].valueX);			
                        y = cp.getYpos(v[i].valueY);
                        pos.push({"x": x, "y": y});
			
                        if(!i) {
                            d += "M "+x+" "+y;
                            }
                        else{
                            d += " L "+x+" "+y;
                            };

                        if(cp.showPoints) {
                            point = this.cnt.drawCircle({
                                "left"        : x,
                                "top"         : y,
                                "radius"      : cp.pointsRadius,
                                "strokeWidth" : 0,
                                "fill"        : color
                                }, layer);
                            this.cnt.addElement("title", point, null, "x: " + v[i].valueX + "\ny: " + v[i].valueY);
                            };

                        };
                    if(cp.showLine) {
			if(cp.fillChart) {
			    y = cp.getYpos(cp.gridYstartValue);
			    d += " L " + cp.getXpos(v[len - 1].valueX) + " " + y + " L " + cp.getXpos(v[0].valueX) + " " + y + " Z";			    
			    };			
                        this.cnt.drawPath({
                            "data"           :d,
                            "strokeWidth"    :cp.lineWidth,
                            "stroke"         :color,
                            "strokeLinecap"  :"round",
                            "strokeLinejoin" :"bevel",
                            "fill"           : (cp.fillChart)? color : "none",
			    "fillOpacity"    : (cp.fillChart)? cp.fillOpacityChart : 0
                            },layer);
                        };
                    this.drawGridAndAxes();
                    this.cnt.raiseOnTop(layer);
		    },
                "pie": function() {
                    var p = this.propObj, cp = this.chartPropObj,
                        radius = cp.chartScale*Math.min(this.width, this.height)/2,
                        angleSum1 = angleSum2 = 0, cx = this.width/2 + p.marginX,
                        cy = this.height/2 + p.marginY + 5,
                        d = "", p1, p2, pV, pL, difAngle, v = this.values, len = v.length,
                        offsetX = offsetY = 0,
                        getPos = function(angle,r) {
                            return {
                                x: (r || radius)*Math.cos(angle*Math.PI/180),
                                y: (r || radius)*Math.sin(angle*Math.PI/180)
                                };
                            },
                        setText = function(txt,l,t) {
                            return this.cnt.drawText(txt, {
                                "top"       : t,
                                "left"      : l,
                                "size"  : p.fontSize,
                                "weight": p.fontWeight,
                                "family": p.fontFamily
                                })
                            };

                    for(var i=0; i<len; i++)
                        {
                        angleSum2 += v[i].angle;
                        difAngle = angleSum1 + (angleSum2 - angleSum1)/2;
                        offsetX = offsetY = 0;

                        p1 = getPos(angleSum1);
                        p2 = getPos(angleSum2);
                        pV = getPos(difAngle, 0.5*radius);
                        pL = getPos(difAngle, 1.05*radius);

                        d  = "M " + cx + " " + cy;
                        d += " l " + p1.x + " " + p1.y;
                        d += " A " + radius + " " + radius+" "+((v[i].angle>180)? "0 1 1":"0 0 1")+" " + (cx+p2.x) + " " + (cy+p2.y);
                        d += " L " + cx + " " + cy;
                        d += " z";

                        this.cnt.drawPath({
                            "data"        :d,
                            "strokeWidth" :cp.pieBorderWidth,
                            "stroke"      :cp.pieBorderColor,
                            "fill"	  :v[i].color
                            });

                        (function(_this_,elem) {
                            elem.setAttribute("x", cx + pV.x - elem.getBBox().width/2);
                            elem.setAttribute("y", cy + pV.y + elem.getBBox().height/2);
                            })(this,setText.call(this, v[i].value));

                        (function(elem) {
                            if(difAngle >0 && difAngle <= 90) {
                                offsetY = elem.getBBox().height;
                                };
                            if(difAngle >90 && difAngle <= 180) {
                                offsetX = -elem.getBBox().width;
                                offsetY = elem.getBBox().height;
                                };
                            if(difAngle >180 && difAngle <= 270) {
                                offsetX = -elem.getBBox().width;
                                };
                            if(difAngle >270 && difAngle <= 360) {};

                            elem.setAttribute("x", cx + pL.x + offsetX);
                            elem.setAttribute("y", cy + pL.y + offsetY);
                            })(setText.call(this, v[i].label));

                        angleSum1 = angleSum2;
                        };
                    }
                },
            "addData": function(valuesAr, propObj) {
		propObj = propObj || {};
                switch(this.propObj.chartType) {
                    case "bar": {
                        var cp = this.chartPropObj = setDefaultProp(propObj, barChartDefaultProp),
			    p = this.propObj,
                            mx = 3, my = 3,/* margins */
                            color, barWidth,
                            h, v = valuesAr, len = v.length,
			    maxValue = (function(ar) {
                            for(var i=0; i<len; i++)
                                ar.push(v[i][0]);
                            return Math.max.apply(null, ar);
                            })([]), maxW = (this.width - (v.length + 1) * mx)/v.length;

			    if(cp.barWidth > maxW) {
				cp.barWidth = maxW;
                                }
			    else{
				mx = (this.width - (v.length * cp.barWidth))/(v.length + 1);console.log(mx);
				};

                        for(var i=0; i<len; i++) {
                            h = (this.height * (v[i][0]/maxValue));
                            this.values.push({
                                "x"     : p.marginX + (i * cp.barWidth + mx * (i + 1)),
                                "y"     : p.marginY + this.height - h - my,
                                "height": h,
                                "label" : v[i][1],
                                "value" : v[i][0],
                                "color" : getColor.call(this, this.chartPropObj.barColor)
                                });
                            };                                                   
                        break;
                        }
                    case "line": {                                         
                        /* Create chart property object */
                        this.chartPropObj = setDefaultProp(propObj, lineChartDefaultProp);
                        var p = this.propObj,
			    v = valuesAr,
			    cp = this.chartPropObj,
			    len = v.length,
			    _this_ = this,
			    absXdif, absYdif;

                        (function(xAr, yAr, _this_) {
                            for(var i=0; i<len; i++) {
                                xAr.push(v[i][0]);
                                yAr.push(v[i][1]);
                                };
                            cp.xMin = Math.min.apply(null, xAr);
                            cp.xMax = Math.max.apply(null, xAr);
                            cp.yMin = Math.min.apply(null, yAr);
                            cp.yMax = Math.max.apply(null, yAr);
			    absXdif = Math.abs(cp.xMax - cp.xMin);
			    absYdif = Math.abs(cp.yMax - cp.yMin);			    
                            })([],[], this);
			
			cp.gridHeight = this.height - 2 * p.gridYmargin;
			cp.gridWidth  = this.width  - 2 * p.gridXmargin;

			/* X */
			if(cp.spacingX && cp.spacingX <= absXdif) {
			    cp.gridXstartValue = Math.floor(cp.xMin/cp.spacingX) * cp.spacingX;
			    cp.gridXstopValue  = cp.gridXstartValue + Math.ceil((cp.xMax - cp.gridXstartValue)/cp.spacingX) * cp.spacingX;
			    absXdif = Math.abs(cp.gridXstopValue - cp.gridXstartValue);
			    }
			else{
			    cp.spacingXlength = cp.spacingXlength || 10;
			    cp.spacingX = absXdif/cp.spacingXlength;
			    cp.gridXstartValue = cp.xMin;
			    cp.gridXstopValue  = cp.xMax;			    
			    };			
			cp.wspX = cp.gridWidth/absXdif;
			cp.gridsX = Math.round(absXdif/cp.spacingX);
			cp.getXpos = function(v) {
			    return p.marginX + p.gridXmargin + (v - cp.gridXstartValue) * cp.wspX;
			    };
			
			/* Y */   
			if(cp.spacingY && cp.spacingY <= absYdif) {
			    cp.gridYstartValue = Math.floor(cp.yMin/cp.spacingY) * cp.spacingY;
			    cp.gridYstopValue  = cp.gridYstartValue + Math.ceil((cp.yMax - cp.gridYstartValue)/cp.spacingY) * cp.spacingY;
			    absYdif = Math.abs(cp.gridYstopValue - cp.gridYstartValue);
			    }
			else{
			    cp.spacingYlength = cp.spacingYlength || 10;
			    cp.gridYstartValue = cp.yMin;
			    cp.gridYstopValue  = cp.yMax;
			    cp.spacingY = absYdif/cp.spacingYlength;
			    };				
			cp.wspY = cp.gridHeight/absYdif;
			cp.gridsY = Math.round(absYdif/cp.spacingY);			    
			cp.getYpos = function(v) {
			    return (_this_.height + p.marginY - p.gridYmargin) - (v - cp.gridYstartValue) * cp.wspY;
			    };		    

                        for(var i=0; i<len; i++) {
                            this.values.push({
                                "valueX": v[i][0],
                                "valueY": v[i][1],
                                "label" : v[i][2]
                                });
                            };
                        break;
                        }
                    case "pie": {
                        this.chartPropObj = setDefaultProp(propObj, pieChartDefaultProp);

                        var p = this.propObj, v = valuesAr, len = v.length, angle, anglesSum = (function(a) {
                        for(var i=0; i<v.length; i++) {
                            a += v[i][0];
                            };
                        return a;
                        })(0);

                        for(var i=0; i<len; i++) {
                            angle = 360 * (v[i][0]/anglesSum);
                            this.values.push({
                                "angle" : angle,
                                "value" : v[i][0],
                                "label" : v[i][1],
                                "color" : getColor.call(this, this.chartPropObj.pieColor)
                                });
                            };
                        }
                    };
                this.drawChart[this.propObj.chartType].call(this);
                }
            },
        /* Private methods */
        getColor = (function() {
            var lastColor = -1, lastC, cName, resultAr = [], colorsAr = ["#ff0066","#00ff66","#ccff00","#ffcc00", "#ff6600","#00ccff","#7f2aff"];
            return function(/* c:Object|String, ?len:Number */ c, len/* colors length */) {
                len = len||1;
                resultAr = [];
                switch(true) {
                    case (c instanceof Object): {
                        var grdType = c.gradientType||"linear";
                        return (grdType==="linear")? this.cnt.createLinearGradient(c) : this.cnt.createRadialGradient(c);
                        }
                    case (c==="auto"): {
                        while(len--) {
                            lastC = (++lastColor === colorsAr.length)? (lastColor=0):lastColor;
                            resultAr.push(colorsAr[lastC]);
                            };
                        break;
                        }
                    case (c==="random"): {
                        while(len--) {
                            while(1) {
                                i = Math.floor(Math.random()*colorsAr.length);
                                if(lastColor != i && resultAr[0] != colorsAr[i]) {
                                    lastColor = i;
                                    resultAr.push(colorsAr[i]);
                                    break;
                                    };
                                };
                            };
                        break;
                        }
                    default: {
                        return c;
                        }
                    };
                return (resultAr.length>1)? resultAr:resultAr[0];
                }})(),
        prepareMethods = function() {
            for(var i in methods) {
                _class.prototype[i] = methods[i];
                };
            },
        cloneObj = function(obj) {
            var objTmp  = {};
            for(var i in obj) {
                objTmp[i] = obj[i];
                };
            return objTmp;
            },
        isUndef = (function(undef) {
            return function(v) {
                return (v === undef)? true : false;
                };
            })(),
        setDefaultProp = function(propObj, defaultProp) {
            var obj = {};
            for(var i in defaultProp) {
                if(isUndef(propObj[i])) {
                    obj[i] = defaultProp[i];
                    continue;
                    }
                obj[i] = propObj[i];
                };
            return obj;
            },// End private methods
        /* Defaults properties of chart and instance of the class SVGChart */
        defaultProp = {
            /* Master properties */
            "chartType"       : "bar",
            "fontFamily"      : "Tahoma, Arial",
            "fontSize"        : "0.8em",
            "fontColor"       : "#000",
            "fontWeight"      : "normal",
            "backgroundColor" : "#EEE",
            "borderWidth"     : 1,
            "borderColor"     : "#CCC",
            "borderRadius"    : 5,
            "marginX"         : 50,
            "marginY"         : 50,
            "cnt"             : null,
            /* Chart label properties */
            "chartLabel"           : "Chart #",
            "chartLabelFontWeight" : "normal",
            "chartLabelFontSize"   : "1.2em",
            "chartLabelFontColor"  : "#000",
            /* Grid properties */
            "gridColor"       : "#000",
            "gridWidth"       : 1,
            "gridOpacity"     : 0.3,
	    "gridFontSize"    : "0.8em",
	    "gridXmargin"     : 20,
	    "gridYmargin"     : 20,
            /* Axes labels properties */
            "axesXLabel"      : "x",
            "axesYLabel"      : "y",
            "axesColor"       : "#000",
            "axesWidth"       : 4,
            "axesLabelFontSize": "1em",
            },
        lineChartDefaultProp = {
            "lineColor"       : "random",
            "lineWidth"       : 2,
            "showPoints"      : true,
            "pointsRadius"    : 4,
	    "fillChart"	      : false,
	    "fillOpacityChart": 0.2,
            "showLine"        : true,
	    "spacingXlength"  : 10,
	    "spacingYlength"  : 10,	    
            "spacingX"        : 0,
            "spacingY"        : 0,
            "showGrid"        : true,
            "showAxes"	      : true,
            "showAxesLabels"  : true,
	    "gridValuesPrecision": 3
            },
        pieChartDefaultProp = {
            "pieBorderColor" : "black",
            "pieColor"       : "random",
            "pieBorderWidth" : 0.5,
            "chartScale"     : 0.95,
            "showGrid"       : false,
            "showAxes"	     : false,
            "showAxesLabels" : false
            },
        barChartDefaultProp = {
            "barColor"       : "random",
            "barRadius"      : 2,
            "barBorderWidth" : 0.6,
            "barBorderColor" : "black",
            "barWidth"       : 100,
            "showGrid"       : false,
            "showAxes"       : true,
            "showAxesLabels" : true
            };// End defaults property
    /* Class contructor */
    var _class = function(propObj) {
        this.propObj      = setDefaultProp(propObj, defaultProp);// Property of SVGChart class instance
        this.cnt          = new SVGDraw(propObj.cntID);// Reference to SVG document
        this.cntWidth     = this.cnt.cntWidth;// Width of container chart (=background width)
        this.cntHeight    = this.cnt.cntHeight;// Height of container chart (=background height)
        if(!this.cntWidth || !this.cntHeight)
            return false;
        this.width        = this.cntWidth  - this.propObj.marginX*2;// Width of chart space
        this.height       = this.cntHeight - this.propObj.marginY*2;// Height of chart space
        this.bg           = null;// Refernece to background element
        this.chartPropObj = null;// Property of charts
        this.values       = [];// Data to draw
        /* Draw background, axis, grid and labels */
        this.initDraw();console.log(this.propObj);
        };// End class constructor
    /* Inheritance */
    prepareMethods();// End inheritance
    /* Return instance of class */
    return _class;// End code
    })();