package com.packtpub.hibernatesearch.rest;

import javax.servlet.ServletContext;
import javax.ws.rs.GET;
import javax.ws.rs.Path;
import javax.ws.rs.PathParam;
import javax.ws.rs.Produces;
import javax.ws.rs.core.Context;
import javax.ws.rs.core.MediaType;

import org.hibernate.Criteria;
import org.hibernate.FetchMode;
import org.hibernate.Session;
import org.hibernate.criterion.Restrictions;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;

import com.packtpub.hibernatesearch.domain.App;
import com.packtpub.hibernatesearch.startup.MasterNodeInitializer;
import com.packtpub.hibernatesearch.startup.SlaveNodeInitializer;

/**
 * RESTowa usuga zaimplementowana w JAX-RS. Pobiera encj App na podstawie jej ID.  Dziki niej 
 * servlet wyszukujcy moe uy zapyta w oparciu o projekcje by pobrac potrzebne dane z Lucene.
 * Baza danych jest odpytywana wycznie gdy uytkownik nacinie przycisk "Szczegy".  Naciskajc 
 * przycisk uytkownik wywouje AJAXowe danie do tej usugi. Wynikiem jest encja przesana w formacie
 * JSON.
 * 
 * Klasa "RestfulApplication" rejestruje usug i udostpnia j uytkownikom.
 * 
 * Adnotacja @Path definiuje URL, ktry bdzie obsugiwany przez usug. W poczeniu z podstawow ciek
 * zdefiniowan w "RestfulApplication" pena cieka bdzie wyglda np. tak http://localhost:8080/rest/appById/1. 
 * 
 * Adnotacja @Produces instruuje JAX-RS by skonwertowa wynik do formatu JSON.
 */
@Path("/appById/{appId}")
@Produces(MediaType.APPLICATION_JSON)
public class AppResource {
	
	/**
	 * Za pomoc adnotacji @Context Jersey wstrzyknie kontekst aplikacji internetowej, dziki czemu Hibernate
	 * SessionFactory utworzona przez MasterNodeInitializer lub SlaveNodeInitializer moe by uywana.
	 */
	@Context
	ServletContext context;
	
	Logger logger = LoggerFactory.getLogger(AppResource.class);

	/**
	 * Adnotacja @GET powoduje, e ta metoda jest wywoywana zawsze, gdy danie HTTP GET przyjdzie pod 
	 * zarejestrowany URL. 
	 */
	@GET
	public App getAppData( @PathParam("appId") Long appId ) {
		
		// // Zainicjalizuj sesj Hibernate uywajc Hibernate SessionFactory utworzon przez MasterNodeInitializer lub SlaveNodeInitializer.
		String mode = (String) context.getAttribute("mode");
		Session session = mode.equals("master") ? MasterNodeInitializer.openSession() : SlaveNodeInitializer.openSession();
		session.beginTransaction();
		
		// Pobierz encj App na podstawie przesanego ID, uywajc zachannego pobierania. Konwersja do JSONa odbdzie 
		// si po zamkniciu sesji Hibernate... gdybymy uyli leniwego pobierania, konwerter JSONa nie zakoczyby operacji 
		// pomylnie po napotkaniu powizanych obiektw.
		Criteria criteria = session.createCriteria(App.class);
		criteria.add( Restrictions.eq("id", appId) );
		criteria.setFetchMode("supportedDevices", FetchMode.SELECT);
		criteria.setFetchMode("customerReviews", FetchMode.SELECT);
		App app = (App) criteria.uniqueResult();
		
		// Czyszczenie Hibernate
		session.getTransaction().commit();
		session.clear();
		session.close();
		
		return app;
	}
	
}
