package com.packtpub.hibernatesearch.servlet;

import java.io.IOException;
import java.util.List;

import javax.servlet.ServletException;
import javax.servlet.annotation.WebServlet;
import javax.servlet.http.HttpServlet;
import javax.servlet.http.HttpServletRequest;
import javax.servlet.http.HttpServletResponse;

import org.hibernate.Session;
import org.hibernate.search.FullTextSession;
import org.hibernate.search.Search;
import org.hibernate.search.query.dsl.QueryBuilder;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;

import com.packtpub.hibernatesearch.domain.App;
import com.packtpub.hibernatesearch.util.StartupDataLoader;

/**
 * Servlet implementujcy kontroler/model obsugujcy przeszukiwanie i renderowanie wynikw przy pomocy widoku w JSP/JSTL.
 * Adnotacja @Webservlet, dostpna w specyfikacji servletw w wersji 3.0, mapuje ten servlet do URLa "search (np. 
 *"http://localhost:8080/search"). We wczeniejszych wersjach specyfikacji ta konfiguracja znajdowaaby si w pliku "web.xml".
 * Podstawowa logika tej operacji wyszukiwania moe by zrefaktoryzowana w celu adaptacji do aplikacji napisanych w Springu, JSFie
 * lub dowolnym innym Javowym frameworku do aplikacji internetowych.
 */
@SuppressWarnings("serial")
@WebServlet("search")
public class SearchServlet extends HttpServlet {

	/**
	 * W tej metodzie zaimplementowane zostay gwne funkcjonalnoci wyszukiwania dla tego servletu. Metoda jest automatycznie wywoywana
	 * przy kadym wywoaniu dania HTTP POST do zmapowanego adresu URL.
	 */
	@SuppressWarnings("unchecked")
	@Override	
	protected void doPost(HttpServletRequest request, HttpServletResponse response) throws ServletException, IOException {
		
		Logger logger = LoggerFactory.getLogger(SearchServlet.class);
		
		// Pobierz sowa kluczowe uytkownika pobrane ze zmiennych CGI
		String searchString = request.getParameter("searchString");
		logger.info("Received searchString [" + searchString + "]");

		// Rozpocznij sesj Hibernate.
		Session session = StartupDataLoader.openSession();
		
		// Utwrz wrapper Hibernate Search wok czystej sesji Hibernate
		FullTextSession fullTextSession = Search.getFullTextSession(session);

		// Rozpocznij transkacj. W tym przypadku nie jest to niezbdne, ale jest tzw. dobr praktyk.
		fullTextSession.beginTransaction();

		// Utwrz obiekt typu QueryBuilder pochodzcy od Hibernate Search i utwrz go dla odpowiedniego indeksu Lucene (w naszym przypadku indeksu "App").
		QueryBuilder queryBuilder = fullTextSession.getSearchFactory().buildQueryBuilder().forEntity( App.class ).get();
		
		// Uyj QueryBuildera by zbudowa w Lucene zapytanie z uyciem sw kluczowych, porwnujce sowa kluczowe wprowadzone przez uytkownika z polami 
		// name i description w klasie App, pola name powizanych aplikacji oraz pola comments wbudowanych obiektw klasy CustomerReview.
		org.apache.lucene.search.Query luceneQuery = queryBuilder
			.keyword()
			.onFields("name", "description", "supportedDevices.name", "customerReviews.comments")
			.matching(searchString)
			.createQuery();
		org.hibernate.Query hibernateQuery = fullTextSession.createFullTextQuery(luceneQuery, App.class);
		
		List<App> apps = hibernateQuery.list();
		logger.info("Found " + apps.size() + " apps");

		// Odcz wyniki z sesji Hibernate (by unikn niechcianej interakcji midzy warstw prezentacji i Hibernate,
		// gdy powizane urzdzenia lub wbudowane obiekty typu CustomerReview s przegldane).
		fullTextSession.clear();

		// Umie wyniki wyszukiwania w obiekcie dania HTTP. 
		request.setAttribute("apps", apps);

		// Oprnij i zamknij sesj Hibernate.
		fullTextSession.getTransaction().commit();
		session.close();
		
		// Przeka danie HTTP, wraz z wynikami wyszukiwania, do widoku zbudowanego JSP/JSTL w celu wyrenderowania strony z wynikami
		getServletContext().getRequestDispatcher("/WEB-INF/pages/search.jsp").forward(request, response);
	}

	/**
	 * Ta metoda jest wywoywana za kadym razem gdy zostanie wywoane danie HTTP GET do zmapowanego URLa.  Dla naszego servletu nie ma znaczenia, 
	 * czy zostanie wywoany metod POST czy GET. W zwizku z tym przekierowujemy danie do metody "doPost()".
	 */
	@Override
	protected void doGet(HttpServletRequest request, HttpServletResponse response) throws ServletException, IOException {	
		this.doPost(request, response);
	}
	
}
