package com.pragprog.aebook.chat.client;

import java.util.Date;
import java.util.GregorianCalendar;
import java.util.List;

import com.google.gwt.core.client.EntryPoint;
import com.google.gwt.core.client.GWT;
import com.google.gwt.event.dom.client.ClickEvent;
import com.google.gwt.event.dom.client.ClickHandler;
import com.google.gwt.event.dom.client.KeyCodes;
import com.google.gwt.event.dom.client.KeyUpEvent;
import com.google.gwt.event.dom.client.KeyUpHandler;
import com.google.gwt.user.client.Timer;
import com.google.gwt.user.client.rpc.AsyncCallback;
import com.google.gwt.user.client.ui.Button;
import com.google.gwt.user.client.ui.HorizontalPanel;
import com.google.gwt.user.client.ui.Label;
import com.google.gwt.user.client.ui.RootPanel;
import com.google.gwt.user.client.ui.TextArea;
import com.google.gwt.user.client.ui.TextBox;
import com.google.gwt.user.client.ui.VerticalPanel;

/**
 * Klasy punktu wejścia definiują metodę <code>onModuleLoad</code>.
 */
public class Chat implements EntryPoint {

    /**
     * Tworzymy zdalne proxy usług, by komunikować się z usługą Greeting.
     * z usługą Greeting.
     */
    private final ChatServiceAsync chatService = GWT
            .create(ChatService.class);

    private String currentChat;

    private long lastMessageTime;

    private TextArea text;
    
    private String user = null;

    /**
     * Sets up a call and callback to retrieve the list of available chats. When
     * the server responds, this will create the link widgets, and add them to
     * the chatListPanel.
     * 
     * @param chatListPanel
     */
    public void populateChats(final VerticalPanel chatListPanel) {
        chatService.getChats(new AsyncCallback<List<ChatRoom>>() {
            public void onFailure(Throwable caught) {
                chatListPanel.add(new Label("Nie można pobrać informacji o pokojach czatu: " + caught));
            }

            public void onSuccess(List<ChatRoom> chats) {
                for (ChatRoom chat : chats) {
                    Button chatButton = new Button(chat.getName()); 
                    chatListPanel.add(chatButton);
                    Chat.this.setupChatClickHandler(chatButton,
                            chat.getName());
                }
                setCurrentChat(chats.get(0).getName());

            }
        });
    }

    protected void setupChatClickHandler(final Button chatButton, final String chat) {
        chatButton.addClickHandler(new ClickHandler() {
            public void onClick(ClickEvent event) {
                setCurrentChat(chat);
            }
        });
    }
    
    public void setCurrentChat(String chat) {
        System.err.println(">>>>>>> Przestawianie pokoju czatu na: " + chat);
        
        text.setText("Wybrany czat: " + chat + "\n");
        currentChat = chat;
        chatService.getMessages(getCurrentChat(), new MessageListCallback());
    }
    
    /**
     * To jest metoda punktu wejścia.
     */
    public void onModuleLoad() {
        final VerticalPanel mainVert = new VerticalPanel(); 

        final VerticalPanel topPanel = new VerticalPanel(); 
        final HorizontalPanel midPanel = new HorizontalPanel();
        final HorizontalPanel bottomPanel = new HorizontalPanel();
        mainVert.add(topPanel); 
        mainVert.add(midPanel);
        mainVert.add(bottomPanel);

        final Label title = new Label("Czat App Engine"); 
        //final Label subtitle = new Label(new Date().toString());
        final Label subtitle = new Label("Mamy dziś: " + (new Date().getDate()) + "." + (1+(new Date().getMonth())) + "." + (1900 + (new Date().getYear()) + " Godzina:" + (1+(new Date().getHours()))+ ":" + (1+(new Date().getMinutes()))));
        
        title.addStyleName("title");
        topPanel.add(title);
        topPanel.add(subtitle);

        final VerticalPanel chatList = new VerticalPanel(); 
        chatList.setBorderWidth(2);
        final Label chatLabel = new Label("Pokoje czatu");
        chatLabel.addStyleName("bold");
        chatList.add(chatLabel);
        chatList.setWidth("10em");
        populateChats(chatList);
        text = new TextArea();
        text.addStyleName("messages");
        text.setWidth("60em");
        text.setHeight("20em");
        midPanel.add(chatList);
        midPanel.add(text);

        final Label label = new Label("Wprowadź wiadomość:");
        label.addStyleName("bold");
        final TextBox messageBox = new TextBox(); 
        messageBox.setWidth("60em");
        final Button sendButton = new Button("Wyślij");
        bottomPanel.add(label);
        bottomPanel.add(messageBox);
        bottomPanel.add(sendButton);

        setupSendMessageHandlers(sendButton, messageBox); 

        RootPanel.get().add(mainVert); 

        // Aktywuje kontrolkę messageBox
        messageBox.setFocus(true); 
        messageBox.selectAll();

        // Tworzymy nowy obiekt klasy Timer.
        Timer elapsedTimer = new Timer() {
            public void run() {                
                chatService.getMessagesSince(getCurrentChat(), lastMessageTime,
                        new MessageListCallback());
            }
        };
        // Ustalamy częstotliwość wywoływania na pół sekundy (500 milisekund).
        elapsedTimer.scheduleRepeating(1500);
    }

    private void setupSendMessageHandlers(final Button sendButton,
            final TextBox messageBox) {
    	// Tworzymy metody obsługujące komponenty sendButton i nameField
        class SendMessageHandler implements ClickHandler,
                KeyUpHandler {
        	/** Uruchamiany po kliknięciu przez użytkownika przycisku sendButton. */
            public void onClick(ClickEvent event) {
                sendMessageToServer();
            }

            /** Uruchamiany po naciśnięciu dowolnego klawisza, przy zaznaczonym komponencie nameField. */
            public void onKeyUp(KeyUpEvent event) {
                if (event.getNativeKeyCode() == KeyCodes.KEY_ENTER) {
                    sendMessageToServer();
                }
            }

            /** Wysyłamy wiadomość na serwer. */
            private void sendMessageToServer() { 
                ChatMessage chatmsg = new ChatMessage(user,
                        messageBox.getText(), getCurrentChat());
                messageBox.setText("");
                chatService.postMessage(chatmsg,
                        new AsyncCallback<Void>() {
                    public void onFailure(Throwable caught) {
                        Chat.this.addNewMessage(new ChatMessage(
                                "System", "Błąd w trakcie wysyłania wiadomości: " +
                                caught.getMessage(),
                                getCurrentChat()));
                    }

                    public void onSuccess(Void v) {
                        //chatService.getMessagesSince(getCurrentChat(), 
                        //        lastMessageTime,
                        //        new MessageListCallback());
                    	/** [Tłumacz do KM]: Ten fragment powoduje dwukrotne ładowanie się wpisów 
                    	 * w aplikacji czatu, po osadzeniu w chmurze. */
                    }
                });
            }
        }
        SendMessageHandler handler = new SendMessageHandler();
        sendButton.addClickHandler(handler);
        messageBox.addKeyUpHandler(handler);
    }
    
    public String getUser() {
        return user;
    } 

    protected String getCurrentChat() {
        return currentChat;
    }

//START: AddNewMessages    
    protected void addNewMessages(ChatMessageList newMessages) {
        lastMessageTime = newMessages.getTimestamp();
        StringBuilder content = new StringBuilder();
        content.append(text.getText());
        for (ChatMessage cm : newMessages.getMessages()) {
            content.append(renderChatMessage(cm));
        }
        text.setText(content.toString());
    }
//KONIEC: AddNewMessages        
    
    @SuppressWarnings("deprecation")
	protected String renderChatMessage(ChatMessage msg) {
        Date d = new Date(msg.getDate());
        GregorianCalendar greg = new GregorianCalendar();
        String dateStr = (1+d.getHours()) + ":" + (1+d.getMinutes()) + "." + d.getSeconds() + " " + d.getDate() + "." + (1+d.getMonth()) + "." + (1900 + d.getYear()) ;
        return "[Od: " + msg.getSenderName() + " o " + dateStr  + "]: " + msg.getMessage() + "\n";  
    }

    protected void addNewMessage(ChatMessage newMessage) {
        text.setText(text.getText() + renderChatMessage(newMessage));
    }
    
    public class MessageListCallback implements AsyncCallback<ChatMessageList> {

        public void onFailure(Throwable caught) {
        }

        public void onSuccess(ChatMessageList result) {
            addNewMessages(result);
        }        
    }
    
}
