# Copyright 2015 The TensorFlow Authors. All Rights Reserved.
#
# Licensed under the Apache License, Version 2.0 (the "License");
# you may not use this file except in compliance with the License.
# You may obtain a copy of the License at
#
#     http://www.apache.org/licenses/LICENSE-2.0
#
# Unless required by applicable law or agreed to in writing, software
# distributed under the License is distributed on an "AS IS" BASIS,
# WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
# See the License for the specific language governing permissions and
# limitations under the License.
# ==============================================================================

"""Przykład / wzorzec budowania modelu LSTM PTB .

Trenowanie modelu opisanego w:
(Zaremba i inni) Recurrent Neural Network Regularization
http://arxiv.org/abs/1409.2329

Istnieją 3 obsługiwane konfiguracje modeli:
=============================================================
| konfiguracja | epoki | treningowy | walidacyjny | testowy
=============================================================
| mała         |    13 |      37.99 |      121.39 | 115.91
| średnia      |    39 |      48.45 |       86.16 |  82.07
| duża         |    55 |      37.87 |       82.62 |  78.29
Dokładne wyniki mogą się różnić w zależności od losowej inicjalizacji.

Hiperparametry używane w modelu:
- init_scale - początkowa skala wag
- learning_rate - początkowa wartość współczynnika uczenia
- max_grad_norm - maksymalna dopuszczalna norma gradientu
- num_layers - liczba warstw LSTM
- num_steps - liczba rozwijanych kroków LSTM
- hidden_size - liczba jednostek LSTM
- max_epoch - liczba epok przeszkolonych z początkowym współczynnikiem uczenia
- max_max_epoch - całkowita liczba epok do treningu
- keep_prob - prawdopodobieństwo utrzymania wag w warstwie porzucania
- lr_decay - spadek współczynnika uczenia dla każdej epoki po "max_epoch"
- batch_size - rozmiar grupy

Dane wymagane dla tego przykładu znajdują się w katalogu data/
w zbiorze danych PTB na stronie internetowej Tomasa Mikolova:

$ wget http://www.fit.vutbr.cz/~imikolov/rnnlm/simple-examples.tgz
$ tar xvf simple-examples.tgz

Uruchamianie:

$ python ptb_word_lm.py

"""
from __future__ import absolute_import
from __future__ import division
from __future__ import print_function

import inspect
import time

import numpy as np
import tensorflow as tf

import reader

flags = tf.flags
logging = tf.logging

flags.DEFINE_string("save_path", None,
                    "Katalog wyjściowy modelu.")
FLAGS = flags.FLAGS


class PTBInput(object):
  """Dane wejściowe."""

  def __init__(self, config, data, name=None):
    self.batch_size = batch_size = config.batch_size
    self.num_steps = num_steps = config.num_steps
    self.epoch_size = ((len(data) // batch_size) - 1) // num_steps
    self.input_data, self.targets = reader.ptb_producer(
        data, batch_size, num_steps, name=name)


class PTBModel(object):
  """Model PTB."""

  def __init__(self, is_training, config, input_):
    self.input = input_

    batch_size = input_.batch_size
    num_steps = input_.num_steps
    size = config.hidden_size
    vocab_size = config.vocab_size

    # Nieco lepsze wyniki można uzyskać z obciążeniem bramki zapominającej
    # zainicjowanym na 1, ale hiperparametry modelu musiałyby być inne,
    # niż podane w opracowaniu.
    def lstm_cell():
      # W najnowszym kodzie źródłowym TensorFlow (od 27 marca 2017 r.),
      # BasicLSTMCell będzie potrzebować parametru reuse, który niestety
      # nie jest zdefiniowany w TensorFlow 1.0. Aby zachować kompatybilność
      # wsteczną, dodajemy tutaj sprawdzanie argumentów:
      if 'reuse' in inspect.getargspec(
          tf.contrib.rnn.BasicLSTMCell.__init__).args:
        return tf.contrib.rnn.BasicLSTMCell(
            size, forget_bias=0.0, state_is_tuple=True,
            reuse=tf.get_variable_scope().reuse)
      else:
        return tf.contrib.rnn.BasicLSTMCell(
            size, forget_bias=0.0, state_is_tuple=True)
    attn_cell = lstm_cell
    if is_training and config.keep_prob < 1:
      def attn_cell():
        return tf.contrib.rnn.DropoutWrapper(
            lstm_cell(), output_keep_prob=config.keep_prob)
    cell = tf.contrib.rnn.MultiRNNCell(
        [attn_cell() for _ in range(config.num_layers)],
                                    state_is_tuple=True)

    self.initial_state = cell.zero_state(batch_size, tf.float32)

    with tf.device("/cpu:0"):
      embedding = tf.get_variable(
          "embedding", [vocab_size, size], dtype=tf.float32)
      inputs = tf.nn.embedding_lookup(embedding, input_.input_data)

    if is_training and config.keep_prob < 1:
      inputs = tf.nn.dropout(inputs, config.keep_prob)

    outputs = []
    state = self.initial_state
    with tf.variable_scope("RNN"):
      for time_step in range(num_steps):
        if time_step > 0: tf.get_variable_scope().reuse_variables()
        (cell_output, state) = cell(inputs[:, time_step, :], state)
        outputs.append(cell_output)

    output = tf.reshape(tf.stack(axis=1, values=outputs), [-1, size])
    softmax_w = tf.get_variable(
        "softmax_w", [size, vocab_size], dtype=tf.float32)
    softmax_b = tf.get_variable(
        "softmax_b", [vocab_size], dtype=tf.float32)
    logits = tf.matmul(output, softmax_w) + softmax_b

    # Zmiana kształtu logitów na tensor 3D dla straty sekwencyjnej
    logits = tf.reshape(logits, [batch_size, num_steps, vocab_size])

    # użycie społecznościowej funkcji straty sekwencyjnej, wyliczanie średniej dla grup
    loss = tf.contrib.seq2seq.sequence_loss(
        logits,
        input_.targets,
        tf.ones([batch_size, num_steps], dtype=tf.float32),
        average_across_timesteps=False,
        average_across_batch=True
    )

    # aktualizacja zmiennych kosztu
    self.cost = cost = tf.reduce_sum(loss)
    self.final_state = state

    if not is_training:
      return

    self.lr = tf.Variable(0.0, trainable=False)
    tvars = tf.trainable_variables()
    grads, _ = tf.clip_by_global_norm(tf.gradients(cost, tvars),
                                      config.max_grad_norm)
    optimizer = tf.train.GradientDescentOptimizer(self.lr)
    self.train_op = optimizer.apply_gradients(
        zip(grads, tvars),
        global_step=tf.contrib.framework.get_or_create_global_step())

    self.new_lr = tf.placeholder(
        tf.float32, shape=[], name="nowy_wspolczynnik_uczenia")
    self.lr_update = tf.assign(self.lr, self.new_lr)

  def assign_lr(self, session, lr_value):
    session.run(self.lr_update, feed_dict={self.new_lr: lr_value})


class SmallConfig(object):
  """Mała konfiguracja"""
  init_scale = 0.1
  learning_rate = 1.0
  max_grad_norm = 5
  num_layers = 2
  num_steps = 20
  hidden_size = 200
  max_epoch = 4
  max_max_epoch = 13
  keep_prob = 1.0
  lr_decay = 0.5
  batch_size = 20
  vocab_size = 10000


def run_epoch(session, model, eval_op=None, verbose=False):
  """Uruchomienie modelu na dostępnych danych."""
  start_time = time.time()
  costs = 0.0
  iters = 0
  state = session.run(model.initial_state)

  fetches = {
      "cost": model.cost,
      "final_state": model.final_state,
  }
  if eval_op is not None:
    fetches["eval_op"] = eval_op

  for step in range(model.input.epoch_size):
    feed_dict = {}
    for i, (c, h) in enumerate(model.initial_state):
      feed_dict[c] = state[i].c
      feed_dict[h] = state[i].h

    vals = session.run(fetches, feed_dict)
    cost = vals["cost"]
    state = vals["final_state"]

    costs += cost
    iters += model.input.num_steps

    if verbose and step % (model.input.epoch_size // 10) == 10:
      print("%.3f nieokreśloność: %.3f szybkość: %.0f wps" %
            (step * 1.0 / model.input.epoch_size,
             np.exp(costs / iters),
             (iters
              * model.input.batch_size/(time.time() - start_time))))

  return np.exp(costs / iters)

raw_data = reader.ptb_raw_data("./simple-examples/data")
train_data, valid_data, test_data, _ = raw_data

config = SmallConfig()
eval_config = SmallConfig()
eval_config.batch_size = 1
eval_config.num_steps = 1

with tf.Graph().as_default():
  initializer = tf.random_uniform_initializer(-config.init_scale,
                                              config.init_scale)

  with tf.name_scope("Trening"):
    train_input = PTBInput(config=config, data=train_data,
                           name="TrainInput")
    with tf.variable_scope("Model", reuse=None,
                           initializer=initializer):
      m = PTBModel(is_training=True, config=config,
                   input_=train_input)
    tf.summary.scalar("Strata treningowa", m.cost)
    tf.summary.scalar("Współczynnik uczenia", m.lr)

  with tf.name_scope("Walidacja"):
    valid_input = PTBInput(config=config, data=valid_data,
                           name="ValidInput")
    with tf.variable_scope("Model", reuse=True,
                           initializer=initializer):
      mvalid = PTBModel(is_training=False, config=config,
                        input_=valid_input)
    tf.summary.scalar("Strata walidacyjna", mvalid.cost)

  with tf.name_scope("Test"):
    test_input = PTBInput(config=eval_config, data=test_data,
                          name="TestInput")
    with tf.variable_scope("Model", reuse=True,
                           initializer=initializer):
      mtest = PTBModel(is_training=False, config=eval_config,
                       input_=test_input)

  sv = tf.train.Supervisor()
  with sv.managed_session() as session:
    for i in range(config.max_max_epoch):
      lr_decay = config.lr_decay ** max(i + 1 - config.max_epoch, 0.0)
      m.assign_lr(session, config.learning_rate * lr_decay)

      print("Epoka: %d Współczynnik uczenia: %.3f"
            % (i + 1, session.run(m.lr)))
      train_perplexity = run_epoch(session, m, eval_op=m.train_op,
                                   verbose=True)
      print("Epoka: %d Nieokreśloność treningowa: %.3f"
            % (i + 1, train_perplexity))
      valid_perplexity = run_epoch(session, mvalid)
      print("Epoka: %d Nieokreśloność walidacyjna: %.3f"
            % (i + 1, valid_perplexity))

    test_perplexity = run_epoch(session, mtest)
    print("Nieokreśloność testowa: %.3f" % test_perplexity)
