# Copyright 2015 The TensorFlow Authors. All Rights Reserved.
#
# Licensed under the Apache License, Version 2.0 (the "License");
# you may not use this file except in compliance with the License.
# You may obtain a copy of the License at
#
#     http://www.apache.org/licenses/LICENSE-2.0
#
# Unless required by applicable law or agreed to in writing, software
# distributed under the License is distributed on an "AS IS" BASIS,
# WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
# See the License for the specific language governing permissions and
# limitations under the License.
# ==============================================================================

"""Prosty, kompleksowy przykład modelu splotowego w stylu LeNet-5, na zbiorze MNIST.
"""
from __future__ import absolute_import
from __future__ import division
from __future__ import print_function

import argparse
import gzip
import os
import sys
import time

import numpy
from six.moves import urllib
from six.moves import xrange  # pylint: disable=redefined-builtin
import tensorflow as tf

SOURCE_URL = 'http://yann.lecun.com/exdb/mnist/'
WORK_DIRECTORY = 'data'
IMAGE_SIZE = 28
NUM_CHANNELS = 1
PIXEL_DEPTH = 255
NUM_LABELS = 10
VALIDATION_SIZE = 5000  # Rozmiar zbioru walidacyjnego.
SEED = 66478  # Ustaw na None dla generatora pseudolosowego.
BATCH_SIZE = 64
NUM_EPOCHS = 10
EVAL_BATCH_SIZE = 64
EVAL_FREQUENCY = 100  # Liczba kroków między ewaluacjami.


def download(filename):
  """Pobieranie danych ze strony internetowej Yanna, chyba że zostały już pobrane."""
  if not os.path.exists(WORK_DIRECTORY):
    os.makedirs(WORK_DIRECTORY)
  filepath = os.path.join(WORK_DIRECTORY, filename)
  if not os.path.exists(filepath):
    filepath, _ = urllib.request.urlretrieve(SOURCE_URL + filename,
                                             filepath)
    size = os.stat(filepath).st_size
    print('Pomyślnie pobrano', filename, size, 'bajtów.')
  return filepath


def extract_data(filename, num_images):
  """Wyodrębnianie obrazów do tensora 4D [indeks obrazu, y, x, kanały].

  Wartości są przeskalowane z [0, 255] na [-0,5, 0,5].
  """
  print('Wyodrębnianie', filename)
  with gzip.open(filename) as bytestream:
    bytestream.read(16)
    buf = bytestream.read(
        IMAGE_SIZE * IMAGE_SIZE * num_images * NUM_CHANNELS)
    data = numpy.frombuffer(buf, dtype=numpy.uint8).astype(
        numpy.float32)
    # Oryginalne dane składają się z pikseli w zakresie od 0 do 255.
    # Wyśrodkuj dane tak, aby środkiem było zero, a rozpiętość wynosiła 1.
    data = (data - (255/2.0))/255 
    data = data.reshape(num_images, IMAGE_SIZE, IMAGE_SIZE,
                        NUM_CHANNELS)
    return data


def extract_labels(filename, num_images):
  """Wyodrębnianie etykiet do wektora identyfikatorów etykiet int64."""
  print('Wyodrębnianie', filename)
  with gzip.open(filename) as bytestream:
    # Odrzuć nagłówek.
    bytestream.read(8)
    # Wczytaj bajty dla etykiet.
    buf = bytestream.read(num_images)
    labels = numpy.frombuffer(buf, dtype=numpy.uint8).astype(
        numpy.int64)
  return labels


def error_rate(predictions, labels):
  """Zwrócenie wskaźnika błędu na podstawie gęstych predykcji i rzadkich etykiet."""
  return 100.0 - (
      100.0 *
      numpy.sum(numpy.argmax(predictions, 1) == labels) /
      predictions.shape[0])

# Zreplikujemy strukturę modelu podgrafu szkoleniowego, jak również
# podgrafu ewaluacyjnego, współdzieląc przy tym parametry treningowe.
def model(data, train=False):
  """Definicja modelu."""
  # Dwuwymiarowa warstwa splotowa, z uzupełnianiem 'SAME'
  # (tzn. wyjściowa mapa cech ma taki sam rozmiar jak wejście).
  # Zauważ, że {strides} jest tablicą czterowymiarową, której kształt
  # odpowiada układowi danych: [indeks obrazu, y, x, głębokość].
  conv = tf.nn.conv2d(data,
                      conv1_weights,
                      strides=[1, 1, 1, 1],
                      padding='SAME')
  # Obciążenie i nieliniowość skorygowana przez linearyzację.
  relu = tf.nn.relu(tf.nn.bias_add(conv, conv1_biases))
  # Maksymalne łączenie. Specyfikacja rozmiaru jądra {ksize} również odpowiada
  # układowi danych. Mamy tu okno łączące o wielkości 2 i rozmiarze kroku 2.
  pool = tf.nn.max_pool(relu,
                        ksize=[1, 2, 2, 1],
                        strides=[1, 2, 2, 1],
                        padding='SAME')
  conv = tf.nn.conv2d(pool,
                      conv2_weights,
                      strides=[1, 1, 1, 1],
                      padding='SAME')
  relu = tf.nn.relu(tf.nn.bias_add(conv, conv2_biases))
  pool = tf.nn.max_pool(relu,
                        ksize=[1, 2, 2, 1],
                        strides=[1, 2, 2, 1],
                        padding='SAME')
  # Przekształcenie prostopadłościanu mapy cech w macierz 2D,
  # aby wprowadzić ją do w pełni połączonych warstw.
  pool_shape = pool.get_shape().as_list()
  reshape = tf.reshape(
      pool,
      [pool_shape[0], pool_shape[1] * pool_shape[2] * pool_shape[3]])
  # W pełni połączona warstwa. Zwróć uwagę, że operacja "+" powoduje
  # automatyczne rozgłaszanie obciążeń.
  hidden = tf.nn.relu(tf.matmul(reshape, fc1_weights) + fc1_biases)
  # Dodanie 50% porzucania tylko podczas treningu. Porzucanie skaluje również aktywacje,
  # więc przy ewaluacji nie ma konieczności ponownego skalowania.
  if train:
    hidden = tf.nn.dropout(hidden, 0.5, seed=SEED)
  return tf.matmul(hidden, fc2_weights) + fc2_biases



# Pobieranie danych.
train_data_filename = download('train-images-idx3-ubyte.gz')
train_labels_filename = download('train-labels-idx1-ubyte.gz')
test_data_filename = download('t10k-images-idx3-ubyte.gz')
test_labels_filename = download('t10k-labels-idx1-ubyte.gz')

# Wyodrębnianie danych do tablic NumPy.
train_data = extract_data(train_data_filename, 60000)
train_labels = extract_labels(train_labels_filename, 60000)
test_data = extract_data(test_data_filename, 10000)
test_labels = extract_labels(test_labels_filename, 10000)

# Generowanie zbioru walidacyjnego.
validation_data = train_data[:VALIDATION_SIZE, ...]
validation_labels = train_labels[:VALIDATION_SIZE]
train_data = train_data[VALIDATION_SIZE:, ...]
train_labels = train_labels[VALIDATION_SIZE:]

num_epochs = NUM_EPOCHS
train_size = train_labels.shape[0]

# Tutaj próbki szkoleniowe i etykiety są wprowadzane do grafu.
# Te węzły zastępcze będą zasilane grupami danych treningowych na każdym
# kroku treningu przy użyciu argumentu {feed_dict} w wywołaniu Run() poniżej.
train_data_node = tf.placeholder(
    tf.float32,
    shape=(BATCH_SIZE, IMAGE_SIZE, IMAGE_SIZE, NUM_CHANNELS))
train_labels_node = tf.placeholder(tf.int64, shape=(BATCH_SIZE,))
eval_data = tf.placeholder(
    tf.float32,
    shape=(EVAL_BATCH_SIZE, IMAGE_SIZE, IMAGE_SIZE, NUM_CHANNELS))

# Poniższe zmienne przechowują wszystkie wagi do wyszkolenia. Są one przekazywane
# jako wartość początkowa, która zostanie przypisana podczas wywołania:
# {tf.global_variables_initializer().run()}
conv1_weights = tf.Variable(
    # filtr 5x5, głębokość 32.
    tf.truncated_normal([5, 5, NUM_CHANNELS, 32],  
                        stddev=0.1,
                        seed=SEED, dtype=tf.float32))
conv1_biases = tf.Variable(tf.zeros([32], dtype=tf.float32))
conv2_weights = tf.Variable(tf.truncated_normal(
    [5, 5, 32, 64], stddev=0.1,
    seed=SEED, dtype=tf.float32))
conv2_biases = tf.Variable(tf.constant(0.1, shape=[64],
                           dtype=tf.float32))
fc1_weights = tf.Variable(  # w pełni połączona, głębokość 512
    tf.truncated_normal([IMAGE_SIZE // 4 * IMAGE_SIZE // 4 * 64, 512],
                        stddev=0.1,
                        seed=SEED,
                        dtype=tf.float32))
fc1_biases = tf.Variable(tf.constant(0.1, shape=[512],
                         dtype=tf.float32))
fc2_weights = tf.Variable(tf.truncated_normal([512, NUM_LABELS],
                                              stddev=0.1,
                                              seed=SEED,
                                              dtype=tf.float32))
fc2_biases = tf.Variable(tf.constant(
    0.1, shape=[NUM_LABELS], dtype=tf.float32))

# Obliczenia treningowe: logity + strata entropii krzyżowej.
logits = model(train_data_node, True)
loss = tf.reduce_mean(tf.nn.sparse_softmax_cross_entropy_with_logits(
    labels=train_labels_node, logits=logits))

# Regularyzacja L2 dla w pełni połączonych parametrów.
regularizers = (tf.nn.l2_loss(fc1_weights)
                + tf.nn.l2_loss(fc1_biases)
                + tf.nn.l2_loss(fc2_weights)
                + tf.nn.l2_loss(fc2_biases))
# Dodanie pojęcia regularyzacji do straty.
loss += 5e-4 * regularizers

# Optymalizator: ustaw zmienną, która jest zwiększana raz na grupę
# i kontroluje rozkład współczynnika uczenia.
batch = tf.Variable(0, dtype=tf.float32)
# Rozkład raz na epokę, przy użyciu schematu wykładniczego zaczynającego się od 0,01.
learning_rate = tf.train.exponential_decay(
    0.01,                # Bazowy wskaźnik uczenia.
    batch * BATCH_SIZE,  # Aktualny indeks do zbioru danych.
    train_size,          # Krok rozkładu.
    0.95,                # Współczynnik rozkładu.
    staircase=True)
# Wykorzystanie prostego pędu do optymalizacji.
optimizer = tf.train.MomentumOptimizer(learning_rate,
                                       0.9).minimize(loss,
                                                     global_step=batch)

# Predykcje dla aktualnej minigrupy treningowej.
train_prediction = tf.nn.softmax(logits)

# Predykcje dla testu i walidacji, które będziemy obliczać rzadziej.
eval_prediction = tf.nn.softmax(model(eval_data))

# Niewielka funkcja narzędziowa do oceny zbioru danych poprzez podawanie
# grup danych do {eval_data} i wyciąganie wyników z {eval_predictions}.
# Oszczędza pamięć i umożliwia działanie na słabszych procesorach.
def eval_in_batches(data, sess):
  """Uzyskanie przewidywań dla zbioru danych przez uruchamianie go w małych grupach."""
  size = data.shape[0]
  if size < EVAL_BATCH_SIZE:
    raise ValueError("rozmiar grupy do ewaluacji większy niż zbiór danych: %d"
                     % size)
  predictions = numpy.ndarray(shape=(size, NUM_LABELS),
                              dtype=numpy.float32)
  for begin in xrange(0, size, EVAL_BATCH_SIZE):
    end = begin + EVAL_BATCH_SIZE
    if end <= size:
      predictions[begin:end, :] = sess.run(
          eval_prediction,
          feed_dict={eval_data: data[begin:end, ...]})
    else:
      batch_predictions = sess.run(
          eval_prediction,
          feed_dict={eval_data: data[-EVAL_BATCH_SIZE:, ...]})
      predictions[begin:, :] = batch_predictions[begin - size:, :]
  return predictions

# Utworzenie sesji lokalnej w celu uruchomienia treningu.
start_time = time.time()
with tf.Session() as sess:
  # Uruchomienie wszystkich inicjalizatorów w celu przygotowania
  # parametrów treningowych.
  tf.global_variables_initializer().run()
  # Pętla przez kroki treningowe.
  for step in xrange(int(num_epochs * train_size) // BATCH_SIZE):
    # Obliczanie przesunięcia aktualnej minigrupy w danych.
    # Zauważ, że możemy użyć lepszej randomizacji w epokach.
    offset = (step * BATCH_SIZE) % (train_size - BATCH_SIZE)
    batch_data = train_data[offset:(offset + BATCH_SIZE), ...]
    batch_labels = train_labels[offset:(offset + BATCH_SIZE)]
    # Ten słownik odwzorowuje dane grupy (jako tablicę NumPy)
    # do węzła na grafie, do którego powinny być przekazane.
    feed_dict = {train_data_node: batch_data,
                 train_labels_node: batch_labels}
    # Uruchomienie optymalizatora w celu aktualizacji wag.
    sess.run(optimizer, feed_dict=feed_dict)
    # kilka dodatkowych informacji po osiągnięciu częstotliwości ewaluacji
    if step % EVAL_FREQUENCY == 0:
      # pobieranie danych z niektórych dodatkowych węzłów
      l, lr, predictions = sess.run([loss, learning_rate,
                                     train_prediction],
                                    feed_dict=feed_dict)
      elapsed_time = time.time() - start_time
      start_time = time.time()
      print('Krok %d (epoka %.2f), %.1f ms' %
            (step, float(step) * BATCH_SIZE / train_size,
             1000 * elapsed_time / EVAL_FREQUENCY))
      print('Strata minigrupy: %.3f, współczynnik uczenia: %.6f' % (l, lr))
      print('Błąd minigrupy: %.1f%%'
            % error_rate(predictions, batch_labels))
      print('Błąd walidacji: %.1f%%' % error_rate(
          eval_in_batches(validation_data, sess), validation_labels))
      sys.stdout.flush()
  # Ostateczny wynik
  test_error = error_rate(eval_in_batches(test_data, sess),
                          test_labels)
  print('Błąd testowy: %.1f%%' % test_error)
