"""Funkcje użytkowe i stałe.

Mam pewne problemy z zapisywaniem w pamięci podręcznej i pobieraniem. 
Gdy używam dekoratora dla pamięci podręcznej, mam błąd z biblioteki streamlit związany z metodą init.
"""
import logging
import pathlib
from typing import Any

from langchain.document_loaders import (
    PyPDFLoader,
    TextLoader,
    UnstructuredEPubLoader,
    UnstructuredWordDocumentLoader,
)
from langchain.memory import ConversationBufferMemory
from langchain.schema import Document


def init_memory():
    """Inicjalizacja pamięci na potrzeby prowadzenia konwersacji.

    Zapisujemy to w pamięci podręcznej, więc tego nie usuwamy
     prz ykązdym restarcie serwera.
     """
    return ConversationBufferMemory(
        memory_key='chat_history',
        return_messages=True,
        output_key='answer'
    )


MEMORY = init_memory()


class EpubReader(UnstructuredEPubLoader):
    def __init__(self, file_path: str | list[str], **unstructured_kwargs: Any):
        super().__init__(file_path, **unstructured_kwargs, mode="elements", strategy="fast")


class DocumentLoaderException(Exception):
    pass


class DocumentLoader(object):
    """Wczytywanie dokumentu w obsługiwanym formacie."""
    supported_extensions = {
        ".pdf": PyPDFLoader,
        ".txt": TextLoader,
        ".epub": EpubReader,
        ".docx": UnstructuredWordDocumentLoader,
        ".doc": UnstructuredWordDocumentLoader
    }


def load_document(temp_filepath: str) -> list[Document]:
    """Wczytaj plik i zwróć go w postaci listy dokumentów.

    W tym momencie nie jest obsługiwanych zbyt wiele błędów.
    """
    ext = pathlib.Path(temp_filepath).suffix
    loader = DocumentLoader.supported_extensions.get(ext)
    if not loader:
        raise DocumentLoaderException(
            f"Nieobsługiwany format {ext}, nie można wczytać tego pliku"
        )

    loaded = loader(temp_filepath)
    docs = loaded.load()
    logging.info(docs)
    return docs
