module MyApplication

open WebSharper
open WebSharper.Sitelets
open WebSharper.UI.Next
open WebSharper.UI.Next.Server

type EndPoint =
    | [<EndPoint "/">] Home
    | [<EndPoint "/authenticated">] Authenticated
    | [<EndPoint "/login">] Login of EndPoint option
    | [<EndPoint "/logout">] Logout

module Site =
    open WebSharper.UI.Next.Html

    module Pages =
        /// Funkcja pomocnicza tworząca nowy adres URL z losowym parametrem
        /// Gwarantuje to, że przeglądarka nie wyświetli wersji zapisanej w pamięci podręcznej
        let R url =
            url + "?d=" + System.Uri.EscapeUriString (System.DateTime.Now.ToString())

        let Links (ctx: Context<_>) =
            let ( => ) title ep = aAttr [attr.href (ctx.Link ep)] [text title]
            let user = ctx.UserSession.GetLoggedInUser() |> Async.RunSynchronously
            ul [
                li ["Strona główna" => EndPoint.Home]
                li ["Wymaga uwierzytelnienia" => EndPoint.Authenticated]
                (if user.IsNone then
                    li ["Logowanie" => EndPoint.Login None]
                else
                    li ["Wylogowanie" => EndPoint.Logout]) 
            ]

        let Home ctx =
            Content.Page(
                Title = "Strona główna",
                Body = [Links ctx; h1 [text "Strona główna. Użyj odsyłaczy."]])

        let Authenticated ctx =
            Content.Page(
                Title = "Wymaga uwierzytelnienia",
                Body = [Links ctx; h1 [text "Ta strona wymaga logowania!"]])

        let Logout ctx =
            Content.Page(
                Title = "Wylogowanie",
                Body = [Links ctx; h1 [text "Nastąpiło wylogowanie."]])

        let Login ctx endpoint =
            let redirectUrl =
                match endpoint with
                | None -> EndPoint.Home
                | Some ep -> ep
                |> ctx.Link
                |> R
            Content.Page(
                Title = "Logowanie",
                Body = [
                    h1 [text "Logowanie"]
                    p [text "I w magiczny sposób nastąpiło zalogowanie."]
                    aAttr [attr.href redirectUrl] [text "Przejdź dalej"]
                ])

    let Authenticated =
        let filter : Sitelet.Filter<EndPoint> =
            { VerifyUser = fun _ -> true
              LoginRedirect = Some >> EndPoint.Login }

        Sitelet.Protect filter
            (Sitelet.Content "/authenticated" EndPoint.Authenticated Pages.Authenticated)

    [<Website>]
    let Main =
        Authenticated
        <|> Application.MultiPage (fun ctx endpoint ->
            match endpoint with
            | EndPoint.Home -> Pages.Home ctx
            | EndPoint.Login endpoint ->
                async {
                    // Logowanie się jako gość bez zgłaszania żądań
                    do! ctx.UserSession.LoginUser "visitor"
                    return! Pages.Login ctx endpoint
                }
            | EndPoint.Logout ->
                async {
                    // Wylogowywanie gościa i kierowanie go do strony głównej
                    do! ctx.UserSession.Logout ()
                    return! Pages.Logout ctx
                }
            | EndPoint.Authenticated -> Content.ServerError
        )
