// Nazwa pliku: ExtremeC_examples_chapter18_2.c
// Opis: Użycie muteksu współdzielonego do ochrony licznika
//              na podstawie pamięci współdzielonej.

#include <stdio.h>
#include <stdint.h>
#include <stdlib.h>
#include <unistd.h>
#include <fcntl.h>
#include <errno.h>
#include <string.h>
#include <sys/mman.h>
#include <sys/wait.h>
#include <pthread.h> // Plik nagłówkowy wymagany przez funkcje mutex_* biblioteki pthread.

#define SHARED_MEM_SIZE 4

// Współdzielony deskryptor pliku używany w celu odwołania się do obiektu pamięci współdzielonej.
int shared_fd = -1;

// Współdzielony deskryptor pliku używany w celu odwołania się
// do obiektu muteksu współdzielonego.
int mutex_shm_fd = -1;

// Wskaźnik prowadzący do licznika współdzielonego.
int32_t* counter = NULL;

// Wskaźnik prowadzący do muteksu współdzielonego.
pthread_mutex_t* mutex = NULL;

void init_control_mechanism() {
  // Otworzenie pamięci muteksu współdzielonego.
  mutex_shm_fd = shm_open("/mutex0", O_CREAT | O_RDWR, 0600);
  if (mutex_shm_fd < 0) {
    fprintf(stderr, "BŁĄD! Nie udało się utworzyć obiektu pamięci współdzielonej: %s\n"
        , strerror(errno));
    exit(1);
  }
  // Alokacja i zmniejszenie obszaru pamięci muteksu współdzielonego.
  if (ftruncate(mutex_shm_fd, sizeof(pthread_mutex_t)) < 0) {
    fprintf(stderr, "BŁĄD! Próba zmniejszenia muteksu zakończyła się niepowodzeniem: %s\n",
        strerror(errno));
    exit(1);
  }
  // Mapowanie pamięci muteksu współdzielonego.
  void* map = mmap(0, sizeof(pthread_mutex_t),
          PROT_READ | PROT_WRITE, MAP_SHARED, mutex_shm_fd, 0);
  if (map == MAP_FAILED) {
    fprintf(stderr, "BŁĄD! Mapowanie zakończyło się niepowodzeniem: %s\n",
            strerror(errno));
    exit(1);
  }
  mutex = (pthread_mutex_t*)map;
  // Inicjalizacja obiektu muteksu.
  int ret = -1;
  pthread_mutexattr_t attr;
  if ((ret = pthread_mutexattr_init(&attr))) {
    fprintf(stderr, "BŁĄD! Próba inicjalizacji atrybutów muteksu zakończyła się niepowodzeniem: %s\n",
        strerror(ret));
    exit(1);
  }
  if ((ret = pthread_mutexattr_setpshared(&attr,
                  PTHREAD_PROCESS_SHARED))) {
    fprintf(stderr, "ŁĄD! Próba zdefiniowania atrybutów muteksu zakończyła się niepowodzeniem: %s\n",
        strerror(ret));
    exit(1);
  }
  if ((ret = pthread_mutex_init(mutex, &attr))) {
    fprintf(stderr, "BŁĄD! Inicjalizacja muteksu zakończyła się niepowodzeniem: %s\n",
        strerror(ret));
    exit(1);
  }
  if ((ret = pthread_mutexattr_destroy(&attr))) {
    fprintf(stderr, "BŁĄD! Próba usunięcia atrybutów muteksu zakończyła się niepowodzeniem: %s\n"
        , strerror(ret));
    exit(1);
  }
}

void shutdown_control_mechanism() {
  int ret = -1;
  if ((ret = pthread_mutex_destroy(mutex))) {
    fprintf(stderr, "BŁĄD! Usunięcie muteksu zakończyło się niepowodzeniem: %s\n",
        strerror(ret));
    exit(1);
  }
  if (munmap(mutex, sizeof(pthread_mutex_t)) < 0) {
    fprintf(stderr, "BŁĄD! Usunięcie mapowania zakończyło się niepowodzeniem: %s\n",
        strerror(errno));
    exit(1);
  }
  if (close(mutex_shm_fd) < 0) {
    fprintf(stderr, "BŁĄD! Zamknięcie muteksu zakończyło się niepowodzeniem: %s\n",
        strerror(errno));
    exit(1);
  }
  if (shm_unlink("/mutex0") < 0) {
    fprintf(stderr, "BŁĄD! Zwolnienie muteksu zakończyło się niepowodzeniem: %s\n",
        strerror(errno));
    exit(1);
  }
}

void init_shared_resource() {
  // Otworzenie obiektu pamięci współdzielonej.
  shared_fd = shm_open("/shm0", O_CREAT | O_RDWR, 0600);
  if (shared_fd < 0) {
    fprintf(stderr, "BŁĄD! Nie udało się utworzyć obiektu pamięci współdzielonej: %s\n"
        , strerror(errno));
    exit(1);
  }
  fprintf(stdout, "Obiekt pamięci współdzielonej został utworzony z deskryptorem pliku o wartości: %d\n",
          shared_fd);
}

void shutdown_shared_resource() {
  if (shm_unlink("/shm0") < 0) {
    fprintf(stderr, "BŁĄD! Nie udało się zwolnić semafora: %s\n",
        strerror(errno));
    exit(1);
  }
}

void inc_counter() {
  usleep(1);
  pthread_mutex_lock(mutex); // Należy sprawdzić wartość zwrotną. value.
  int32_t temp = *counter;
  usleep(1);
  temp++;
  usleep(1);
  *counter = temp;
  pthread_mutex_unlock(mutex); // Należy sprawdzić wartość zwrotną. value.
  usleep(1);
}

int main(int argc, char** argv) {

  // Proces nadrzędny musi zainicjalizować zasób współdzielony.
  init_shared_resource();

  // Proces nadrzędny musi zainicjalizować mechanizm kontroli.
  init_control_mechanism();

  // Alokacja i zmniejszenie obszaru pamięci współdzielonej.
  if (ftruncate(shared_fd, SHARED_MEM_SIZE * sizeof(char)) < 0) {
    fprintf(stderr, "BŁĄD! Próba zmniejszenia obszaru pamięci zakończyła się niepowodzeniem: %s\n",
            strerror(errno));
    return 1;
  }
  fprintf(stdout, "Obszar pamięci został zmniejszony.\n");

  // Mapowanie pamięci współdzielonej i inicjalizacja wskaźnika.
  void* map = mmap(0, SHARED_MEM_SIZE, PROT_WRITE,
          MAP_SHARED, shared_fd, 0);
  if (map == MAP_FAILED) {
    fprintf(stderr, "BŁĄD! Mapowanie zakończyło się niepowodzeniem: %s\n",
            strerror(errno));
    return 1;
  }
  counter = (int32_t*)map;
  *counter = 0;

  // Utworzenie nowego procesu.
  pid_t pid = fork();
  if (pid) { // Proces nadrzędny.
    // Inkrementacja licznika.
    inc_counter();
    fprintf(stdout, "Dla procesu potomnego licznik ma wartość %d.\n",
        *counter);

    // Oczekiwanie na zakończenie działania procesu potomnego.
    int status = -1;
    wait(&status);
    fprintf(stdout, "Proces potomny zakończył działanie i zwrócił wartość %d.\n",
        status);
  } else { // Proces potomny.
    // Inkrementacja licznika.
    inc_counter();
    fprintf(stdout, "Dla procesu potomnego licznik ma wartość %d.\n",
        *counter);
  }
  if (munmap(counter, SHARED_MEM_SIZE) < 0) {
    fprintf(stderr, "BŁĄD! Usunięcie mapowania zakończyło się niepowodzeniem: %s\n",
            strerror(errno));
    return 1;
  }
  if (close(shared_fd) < 0) {
    fprintf(stderr, "BŁĄD! Zamknięcie obiektu pamięci współdzielonej z zakończyło się wartością: %s\n",
        strerror(errno));
    return 1;
  }

  // Tylko proces nadrzędny może zamknąć zasób współdzielony
  // i wykorzystany mechanizm kontroli.
  if (pid) {
    shutdown_shared_resource();
    shutdown_control_mechanism();
  }

  return 0;
}
