// Nazwa pliku: ExtremeC_examples_chapter18_4_shared_mem.c
// Opis: Definicje funkcji znajdujących się w klasie pamięci współdzielonej.

#include <stdio.h>
#include <stdlib.h>
#include <string.h>
#include <unistd.h>
#include <errno.h>
#include <fcntl.h>
#include <sys/mman.h>

#define TRUE 1
#define FALSE 0

typedef int32_t bool_t;

bool_t owner_process_set = FALSE;
bool_t owner_process = FALSE;

typedef struct {
  char* name;
  int shm_fd;
  void* map_ptr;
  char* ptr;
  size_t size;
} shared_mem_t;

shared_mem_t* shared_mem_new() {
  return (shared_mem_t*)malloc(sizeof(shared_mem_t));
}

void shared_mem_delete(shared_mem_t* obj) {
  free(obj->name);
  free(obj);
}

void shared_mem_ctor(shared_mem_t* obj, const char* name,
        size_t size) {
  obj->size = size;
  obj->name = (char*)malloc(strlen(name) + 1);
  strcpy(obj->name, name);
  obj->shm_fd = shm_open(obj->name, O_RDWR, 0600);
  if (obj->shm_fd >= 0) {
    if (!owner_process_set) {
      owner_process = FALSE;
      owner_process_set = TRUE;
    }
    printf("Pamięć współdzielona %s została otworzona.\n", obj->name);
  } else if (errno == ENOENT) {
    printf("OSTRZEŻENIE: Pamięć współdzielona %s nie istnieje.\n",
            obj->name);
    obj->shm_fd = shm_open(obj->name,
            O_CREAT | O_RDWR, 0600);
    if (obj->shm_fd >= 0) {
      if (!owner_process_set) {
        owner_process = TRUE;
        owner_process_set = TRUE;
      }
      printf("Pamięć współdzielona %s została utworzona i otworzona.\n",
              obj->name);
      if (ftruncate(obj->shm_fd, obj->size) < 0) {
        fprintf(stderr, "BŁĄD(%s): Zmniejszenie zakończyło się niepowodzeniem: %s\n",
            obj->name, strerror(errno));
        exit(1);
      }
    } else {
      fprintf(stderr,
          "BŁĄD(%s): Nie udało się utworzyć pamięci współdzielonej: %s\n",
          obj->name, strerror(errno));
      exit(1);
    }
  } else {
      fprintf(stderr,
          "BŁĄD(%s): Nie udało się utworzyć pamięci współdzielonej: %s\n",
          obj->name, strerror(errno));
    exit(1);
  }
  obj->map_ptr = mmap(0, obj->size, PROT_READ | PROT_WRITE,
      MAP_SHARED, obj->shm_fd, 0);
  if (obj->map_ptr == MAP_FAILED) {
    fprintf(stderr, "BŁĄD(%s): Mapowanie pamięci współdzielonej zakończyło się niepowodzeniem: %s\n",
        name, strerror(errno));
    exit(1);
  }
  obj->ptr = (char*)obj->map_ptr;
}

void shared_mem_dtor(shared_mem_t* obj) {
  if (munmap(obj->map_ptr, obj->size) < 0) {
    fprintf(stderr, "BŁĄD(%s): Mapowanie zakończyło się niepowodzeniem: %s\n",
        obj->name, strerror(errno));
    exit(1);
  }
  printf("Pamięć współdzielona %s nie jest mapowana.\n", obj->name);
  if (close(obj->shm_fd) < 0) {
    fprintf(stderr,
        "BŁĄD(%s): Zwolnienie pamięci współdzielonej zakończyło się niepowodzeniem: %s\n",
        obj->name, strerror(errno));
    exit(1);
  }
  printf("Pamięć współdzielona %s została zamknięta.\n", obj->name);
  if (owner_process) {
    if (shm_unlink(obj->name) < 0) {
      fprintf(stderr,
          "BŁĄD(%s): Usunięcie mapowania zakończyło się niepowodzeniem: %s\n",
          obj->name, strerror(errno));
      exit(1);
    }
    printf("Pamięć współdzielona %s została zwolniona.\n", obj->name);
  }
}

char* shared_mem_getptr(shared_mem_t* obj) {
  return obj->ptr;
}

bool_t shared_mem_isowner(shared_mem_t* obj) {
  return owner_process;
}

void shared_mem_setowner(shared_mem_t* obj, bool_t is_owner) {
    owner_process = is_owner;
}
