// Nazwa pliku: ExtremeC_examples_chapter18_3.c
// Opis: Pokazuje użycie pamięci współdzielonej razem z muteksem
//              współdzielonym w celu przeprowadzenia operacji
//              anulowania liczby w procesie.

#include <stdio.h>
#include <stdint.h>
#include <stdlib.h>
#include <unistd.h>
#include <fcntl.h>
#include <errno.h>
#include <string.h>
#include <sys/mman.h>
#include <sys/wait.h>
#include <signal.h>
#include <pthread.h> // Plik nagłówkowy wymagany przez funkcje mutex_* biblioteki pthread.

typedef uint16_t bool_t;

#define TRUE 1
#define FALSE 0

#define MUTEX_SHM_NAME "/mutex0"
#define SHM_NAME "/shm0"

// Współdzielony deskryptor pliku używany w celu odwołania się do obiektu
// pamięci współdzielonej zawierającego opcję anulowania.
int cancel_flag_shm_fd = -1;

// Opcja określająca, czy bieżący proces jest właścicielem
// obiektu pamięci współdzielonej.
bool_t cancel_flag_shm_owner = FALSE;

// Współdzielony deskryptor pliku używany w celu odwołania się
// do obiektu muteksu współdzielonego.
int mutex_shm_fd = -1;

// Muteks współdzielony.
pthread_mutex_t* mutex = NULL;

// Opcja określająca, czy bieżący proces jest właścicielem
// obiektu pamięci współdzielonej.
bool_t mutex_owner = FALSE;

// Wskaźnik prowadzący do opcji anulowania przechowywanej w pamięci współdzielonej.
bool_t* cancel_flag = NULL;

void init_shared_resource() {
  // Otworzenie obiektu pamięci współdzielonej.
  cancel_flag_shm_fd = shm_open(SHM_NAME, O_RDWR, 0600);
  if (cancel_flag_shm_fd >= 0) {
    cancel_flag_shm_owner = FALSE;
    fprintf(stdout, "Obiekt pamięci współdzielonej został otworzony.\n");
  } else if (errno == ENOENT) {
    fprintf(stderr,
            "OSTRZEŻENIE: Obiekt pamięci współdzielonej nie istnieje.\n");
    fprintf(stdout, "Tworzenie obiektu pamięci współdzielonej ...\n");
    cancel_flag_shm_fd = shm_open(SHM_NAME,
            O_CREAT | O_EXCL | O_RDWR, 0600);
    if (cancel_flag_shm_fd >= 0) {
      cancel_flag_shm_owner = TRUE;
      fprintf(stdout, "Obiekt pamięci współdzielonej został utworzony.\n");
    } else {
      fprintf(stderr,
          "BŁĄD! Nie udało się utworzyć obiektu pamięci współdzielonej: %s\n",
          strerror(errno));
      exit(1);
    }
  } else {
      fprintf(stderr,
          "BŁĄD! Nie udało się utworzyć obiektu pamięci współdzielonej: %s\n",
          strerror(errno));
    exit(1);
  }
  if (cancel_flag_shm_owner) {
    // Alokacja i zmniejszenie obszaru pamięci współdzielonej.
    if (ftruncate(cancel_flag_shm_fd, sizeof(bool_t)) < 0) {
      fprintf(stderr, "BŁĄD! Próba zmniejszenia obszaru pamięci zakończyła się niepowodzeniem: %s\n",
              strerror(errno));
      exit(1);
    }
    fprintf(stdout, "Obszar pamięci został zmniejszony.\n");
  }
  // Mapowanie pamięci współdzielonej i inicjalizacja opcji anulowania.
  void* map = mmap(0, sizeof(bool_t), PROT_WRITE, MAP_SHARED,
      cancel_flag_shm_fd, 0);
  if (map == MAP_FAILED) {
    fprintf(stderr, "BŁĄD! Mapowanie zakończyło się niepowodzeniem: %s\n",
            strerror(errno));
    exit(1);
  }
  cancel_flag = (bool_t*)map;
  if (cancel_flag_shm_owner) {
    *cancel_flag = FALSE;
  }
}

void shutdown_shared_resource() {
  if (munmap(cancel_flag, sizeof(bool_t)) < 0) {
    fprintf(stderr, "BŁĄD! Usunięcie mapowania zakończyło się niepowodzeniem: %s\n",
            strerror(errno));
    exit(1);
  }
  if (close(cancel_flag_shm_fd) < 0) {
    fprintf(stderr,
        "BŁĄD! Zwolnienie pamięci współdzielonej: %s\n",
        strerror(errno));
    exit(1);
  }
  if (cancel_flag_shm_owner) {
    sleep(1);
    if (shm_unlink(SHM_NAME) < 0) {
      fprintf(stderr,
          "BŁĄD! Usunięcie mapowania zakończyło się niepowodzeniem: %s\n",
          strerror(errno));
      exit(1);
    }
  }
}

void init_control_mechanism() {
  // Otworzenie pamięci muteksu współdzielonego.
  mutex_shm_fd = shm_open(MUTEX_SHM_NAME, O_RDWR, 0600);
  if (mutex_shm_fd >= 0) {
    // Istnieje muteks obiektu pamięci współdzielonej i jestem jego właścicielem.
    mutex_owner = FALSE;
    fprintf(stdout,
            "Muteks obiektu pamięci współdzielonej został otworzony.\n");
  } else if (errno == ENOENT) {
    fprintf(stderr,
            "OSTRZEŻENIE: Muteks pamięci współdzielonej nie istnieje.\n");
    fprintf(stdout,
            "Tworzenie muteksu obiektu pamięci współdzielonej ...\n");
    mutex_shm_fd = shm_open(MUTEX_SHM_NAME,
            O_CREAT | O_EXCL | O_RDWR, 0600);
    if (mutex_shm_fd >= 0) {
      mutex_owner = TRUE;
      fprintf(stdout,
              "Muteks obiektu pamięci współdzielonej został utworzony.\n");
    } else {
      fprintf(stderr,
          "BŁĄD! Nie udało się utworzyć muteksu obiektu pamięci współdzielonej: %s\n",
          strerror(errno));
      exit(1);
    }
  } else {
    fprintf(stderr,
        "BŁĄD! Nie udało się utworzyć muteksu obiektu pamięci współdzielonej: %s\n",
        strerror(errno));
    exit(1);
  }
  if (mutex_owner) {
    // Alokacja i zmniejszenie obszaru pamięci muteksu współdzielonego.
    if (ftruncate(mutex_shm_fd, sizeof(pthread_mutex_t)) < 0) {
      fprintf(stderr,
          "BŁĄD! Zmniejszenie muteksu zakończyło się niepowodzeniem: %s\n",
          strerror(errno));
      exit(1);
    }
  }
  // Mapowanie pamięci muteksu współdzielonego.
  void* map = mmap(0, sizeof(pthread_mutex_t),
          PROT_READ | PROT_WRITE, MAP_SHARED, mutex_shm_fd, 0);
  if (map == MAP_FAILED) {
    fprintf(stderr, "BŁĄD! Mapowanie zakończyło się niepowodzeniem: %s\n",
            strerror(errno));
    exit(1);
  }
  mutex = (pthread_mutex_t*)map;
  if (mutex_owner) {
    int ret = -1;
    pthread_mutexattr_t attr;
    if ((ret = pthread_mutexattr_init(&attr))) {
      fprintf(stderr,
          "BŁĄD! Inicjalizacja atrybutów muteksu zakończyła się niepowodzeniem: %s\n",
          strerror(ret));
      exit(1);
    }
    if ((ret = pthread_mutexattr_setpshared(&attr,
                    PTHREAD_PROCESS_SHARED))) {
      fprintf(stderr,
          "BŁĄD! Ustawienie atrybutów muteksu zakończyło się niepowodzeniem: %s\n",
          strerror(ret));
      exit(1);
    }
    if ((ret = pthread_mutex_init(mutex, &attr))) {
      fprintf(stderr,
          "BŁĄD! Inicjalizacja muteksu zakończyła się niepowodzeniem: %s\n",
          strerror(ret));
      exit(1);
    }
    if ((ret = pthread_mutexattr_destroy(&attr))) {
      fprintf(stderr,
          "BŁĄD! Usunięcie atrybutów muteksu zakończyło się niepowodzeniem: %s\n",
          strerror(ret));
      exit(1);
    }
  }
}

void shutdown_control_mechanism() {
  sleep(1);
  if (mutex_owner) {
    int ret = -1;
    if ((ret = pthread_mutex_destroy(mutex))) {
      fprintf(stderr,
          "OSTRZEŻENIE: Usunięcie muteksu zakończyło się niepowodzeniem: %s\n",
          strerror(ret));
    }
  }
  if (munmap(mutex, sizeof(pthread_mutex_t)) < 0) {
    fprintf(stderr, "BŁĄD! Usunięcie mapowania zakończyło się niepowodzeniem: %s\n",
        strerror(errno));
    exit(1);
  }
  if (close(mutex_shm_fd) < 0) {
    fprintf(stderr, "BŁĄD! Zamknięcie muteksu zakończyło się niepowodzeniem: %s\n",
        strerror(errno));
    exit(1);
  }
  if (mutex_owner) {
    if (shm_unlink(MUTEX_SHM_NAME) < 0) {
      fprintf(stderr, "BŁĄD! Zwolnienie muteksu zakończyło się niepowodzeniem: %s\n",
          strerror(errno));
      exit(1);
    }
  }
}

bool_t is_canceled() {
  pthread_mutex_lock(mutex); // Należy sprawdzić wartość zwrotną. value
  bool_t temp = *cancel_flag;
  pthread_mutex_unlock(mutex); // Należy sprawdzić wartość zwrotną. value
  return temp;
}

void cancel() {
  pthread_mutex_lock(mutex); // Należy sprawdzić wartość zwrotną. value
  *cancel_flag = TRUE;
  pthread_mutex_unlock(mutex); // Należy sprawdzić wartość zwrotną. value
}

void sigint_handler(int signo) {
  fprintf(stdout, "\nObsługa sygnału INT: %d ...\n", signo);
  cancel();
}

int main(int argc, char** argv) {

  signal(SIGINT, sigint_handler);

  // Proces nadrzędny musi zainicjalizować zasób współdzielony.
  init_shared_resource();

  // Proces nadrzędny musi zainicjalizować mechanizm kontroli.
  init_control_mechanism();

  while(!is_canceled()) {
    fprintf(stdout, "Operacja w toku ...\n");
    sleep(1);
  }

  fprintf(stdout, "Otrzymano sygnał anulowania.\n");

  shutdown_shared_resource();
  shutdown_control_mechanism();

  return 0;
}
